import Promise from 'promise-polyfill';
import {
    IAccount,
    IAdalJsConfig,
    SignInExperienceType,
    AccountType,
    AuthProviderConfigType
} from '@mecontrol/public-api';
import {
    IAuthProvider,
    IAuthOperations,
    ISignInArgs, ISignInToArgs,
    ISignOutFromAppArgs, ISignOutFromIdpArgs, ISignOutAndForgetFromIdpArgs,
    ISwitchArgs, ISwitchToArgs
} from '@mecontrol/common';
import { ME, createError } from '@mecontrol/web-inline';
import { isStringOrFunction } from '../../../utilities';
import { IRememberedAccountsProvider, createAadAccountsProvider } from './webAccountProviders';
import { canLeverageRememberedAccounts } from './operationHelpers';

export function createAdalJsAuthProvider(config: IAdalJsConfig): IAuthProvider {
    return new AdalJsProvider(config);
}

class AdalJsProvider implements IAuthProvider {

    public navProvider: undefined;
    public supportedSignInAccountTypes: SignInExperienceType = SignInExperienceType.Aad;
    private webAadIdpProvider: IRememberedAccountsProvider;

    constructor(private config: IAdalJsConfig) {
        this.webAadIdpProvider = createAadAccountsProvider(config.aad);
    }

    private supportsMsaFed(): boolean {
        return this.config.type === AuthProviderConfigType.AdalJsWithMsaFed;
    }

    public isOperationSupported(authOperation: keyof IAuthOperations, accountType: AccountType): boolean {
        if (accountType !== AccountType.AAD && accountType !== AccountType.MSA_FED) {
            return false;
        }

        switch (authOperation) {
            case "signOutFromIdp":
                return isStringOrFunction(this.config.aad?.signOutUrl);

            case "signOutAndForgetFromIdp":
                return isStringOrFunction(this.config.aad?.signOutAndForgetUrl);

            case "getRememberedAccounts":
                return ME.Config.aad
                    && ME.Config.remAcc
                    && canLeverageRememberedAccounts(this, AccountType.AAD)
                    && isStringOrFunction(this.config.aad?.rememberedAccountsUrl);

            // signIn, signInTo, switch, switchTo, signOutFromApp
            default:
                return typeof this.config[authOperation] === 'function';
        }
    }

    public signIn(args: ISignInArgs): void {
        this.config.signIn(args);
    }

    public signInTo(args: ISignInToArgs): void {
        if (this.isOperationSupported('signInTo', args.nextAccount.type)) {
            this.config.signInTo!(args);
        } else {
            throw createError('signInTo is not supported');
        }
    }

    public signOutFromApp(args: ISignOutFromAppArgs): void {
        this.config.signOutFromApp(args);
    }

    public switch(args: ISwitchArgs): void {
        if (this.isOperationSupported('switch', AccountType.AAD)) {
            this.config.switch!(args);
        } else {
            throw createError('switch is not supported');
        }
    }

    public switchTo(args: ISwitchToArgs): void {
        if (this.isOperationSupported('switchTo', args.nextAccount.type)) {
            this.config.switchTo!(args);
        } else {
            throw createError('switchTo is not supported');
        }    
    }

    public signOutFromIdp(args: ISignOutFromIdpArgs): Promise<void> {
        if (this.isOperationSupported('signOutFromIdp', args.account.type)) {
            return this.webAadIdpProvider.signOutFromIdp(args);
        }

        return Promise.reject(createError('signOutFromIdp is not supported'));
    }

    public signOutAndForgetFromIdp(args: ISignOutAndForgetFromIdpArgs): Promise<void> {
        if (this.isOperationSupported('signOutAndForgetFromIdp', args.account.type)) {
            return this.webAadIdpProvider.signOutAndForgetFromIdp(args);
        }

        return Promise.reject(createError('signOutAndForgetFromIdp is not supported'));
    }

    public getRememberedAccounts(): Promise<IAccount[]> {
        if (this.isOperationSupported('getRememberedAccounts', AccountType.AAD)) {
            let rememberedAccounts = this.webAadIdpProvider.getRememberedAccounts();

            // Filter out MSA_FED accounts if they are not supported by the partner
            if (this.supportsMsaFed()) {
                return rememberedAccounts;
            }
            else {
                return rememberedAccounts
                    .then(accounts => accounts.filter(acc => acc.type !== AccountType.MSA_FED));
            }
        }

        return Promise.reject(createError('getRememberedAccounts is not supported'));
    }
}
