import Promise from 'promise-polyfill';
import {
    IAccount,
    SignInExperienceType,
    AccountType,
    IMsalJsConfig,
    AuthProviderConfigType
} from '@mecontrol/public-api';
import {
    IAuthProvider,
    IAuthOperations,
    ISignInArgs, ISignInToArgs,
    ISignOutFromAppArgs, ISignOutFromIdpArgs, ISignOutAndForgetFromIdpArgs,
    ISwitchArgs, ISwitchToArgs
} from '@mecontrol/common';
import { ME, createError } from '@mecontrol/web-inline';
import { isStringOrFunction } from '../../../utilities';
import { IRememberedAccountsProvider, createAadAccountsProvider } from './webAccountProviders';
import { canLeverageRememberedAccounts } from './operationHelpers';

export function createMsalJsAuthProvider(config: IMsalJsConfig): IAuthProvider {
    return new MsalJsProvider(config);
}

class MsalJsProvider implements IAuthProvider {

    public navProvider: undefined;
    public supportedSignInAccountTypes: SignInExperienceType = SignInExperienceType.Aad;
    private webAadIdpProvider: IRememberedAccountsProvider;

    constructor(private config: IMsalJsConfig) {
        this.webAadIdpProvider = createAadAccountsProvider(config.aad);
    }

    private supportsMsa(): boolean {
        return this.config.type === AuthProviderConfigType.MsalJsWithMsa;
    }

    public isOperationSupported(authOperation: keyof IAuthOperations, accountType: AccountType): boolean {
        switch (authOperation) {
            case "signOutFromIdp":
                return isStringOrFunction(this.config.aad?.signOutUrl);

            case "signOutAndForgetFromIdp":
                return isStringOrFunction(this.config.aad?.signOutAndForgetUrl);

            case "getRememberedAccounts":
                return ME.Config.aad
                    && ME.Config.remAcc
                    && canLeverageRememberedAccounts(this, AccountType.AAD)
                    && isStringOrFunction(this.config.aad?.rememberedAccountsUrl);

            // signIn, signInTo, switch, switchTo, signOutFromApp
            default:
                return typeof this.config[authOperation] === 'function';
        }
    }

    public signIn(args: ISignInArgs): void {
        this.config.signIn(args);
    }

    public signInTo(args: ISignInToArgs): void {
        if (this.isOperationSupported('signInTo', args.nextAccount.type)) {
            this.config.signInTo!(args);
        } else {
            throw createError('signInTo is not supported');
        }
    }

    public signOutFromApp(args: ISignOutFromAppArgs): void {
        this.config.signOutFromApp(args);
    }

    public switch(args: ISwitchArgs): void {
        if (this.isOperationSupported('switch', AccountType.AAD)) {
            this.config.switch!(args);
        } else {
            throw createError('switch is not supported');
        }
    }

    public switchTo(args: ISwitchToArgs): void {
        if (this.isOperationSupported('switchTo', args.nextAccount.type)) {
            this.config.switchTo!(args);
        } else {
            throw createError('switchTo is not supported');
        }
    }

    public signOutFromIdp(args: ISignOutFromIdpArgs): Promise<void> {
        if (this.isOperationSupported('signOutFromIdp', args.account.type)) {
            return this.webAadIdpProvider.signOutFromIdp(args);
        }

        return Promise.reject(createError('signOutFromIdp is not supported'));
    }

    public signOutAndForgetFromIdp(args: ISignOutAndForgetFromIdpArgs): Promise<void> {
        if (this.isOperationSupported('signOutAndForgetFromIdp', args.account.type)) {
            return this.webAadIdpProvider.signOutAndForgetFromIdp(args);
        }

        return Promise.reject(createError('signOutAndForgetFromIdp is not supported'));
    }

    public getRememberedAccounts(): Promise<IAccount[]> {
        if (this.isOperationSupported('getRememberedAccounts', AccountType.AAD)) {
            // Filter out MSA_FED accounts if MSA is supported
            if (this.supportsMsa()) {
                return this.webAadIdpProvider.getRememberedAccounts();
            } else {
                // Filter out both MSA_FED and MSA
                return this.webAadIdpProvider.getRememberedAccounts()
                    .then(accounts => accounts.filter(acc => acc.type === AccountType.AAD));
            }
        }
        return Promise.reject(createError('getRememberedAccounts is not supported'));
    }
}
