import { IAccount, SignInExperienceType, ICustomAuthProviderConfig } from '@mecontrol/public-api';
import { Promise, createError } from '@mecontrol/web-inline';
import {
    IAuthOperations,
    IAuthProvider,
    ISignInArgs,
    ISignInToArgs,
    ISignOutFromAppArgs,
    ISignOutFromIdpArgs,
    ISwitchArgs,
    ISwitchToArgs,
    ISignOutAndForgetFromIdpArgs
} from '@mecontrol/common';

export function createPartnerManagedAuthProvider(config: ICustomAuthProviderConfig): IAuthProvider {
    return new PartnerManagedAuthProvider(config);
}

class PartnerManagedAuthProvider implements IAuthProvider {
    public navProvider: undefined;

    public supportedSignInAccountTypes: SignInExperienceType = this.config.signInExperienceTypes;


    constructor(private config: ICustomAuthProviderConfig) {}

    public isOperationSupported(authOperation: keyof IAuthOperations): boolean {
        return typeof this.config[authOperation] === "function";
    }

    public signIn(args: ISignInArgs): void {
        return this.config.signIn(args);
    }

    public signInTo(args: ISignInToArgs): void {
        if (this.isOperationSupported("signInTo")) {
            return this.config.signInTo!(args);
        }
        else {
            throw createError("switch is not supported");
        }
    }

    public signOutFromApp(args: ISignOutFromAppArgs): void {
        return this.config.signOutFromApp(args);
    }

    public signOutFromIdp(args: ISignOutFromIdpArgs): Promise<void> {
        if (this.isOperationSupported("signOutFromIdp")) {
            return Promise.resolve(this.config.signOutFromIdp!(args));
        }
        else {
            throw createError("switch is not supported");
        }
    }

    public signOutAndForgetFromIdp(args: ISignOutAndForgetFromIdpArgs): Promise<void> {
        if (this.isOperationSupported("signOutFromIdp")) {
            return Promise.resolve(this.config.signOutAndForgetFromIdp!(args));
        }
        else {
            throw createError("switch is not supported");
        }
    }

    public switch(args: ISwitchArgs): void {
        if (this.isOperationSupported("switch")) {
            return this.config.switch!(args);
        }
        else {
            throw createError("switch is not supported");
        }
    }

    public switchTo(args: ISwitchToArgs): void {
        if (this.isOperationSupported("switchTo")) {
            return this.config.switchTo!(args);
        }
        else {
            throw createError("switchTo is not supported");
        }
    }

    public getRememberedAccounts(): Promise<IAccount[]> {
        if (this.isOperationSupported("getRememberedAccounts")) {
            return Promise.resolve(this.config.getRememberedAccounts!());
        }
        else {
            throw createError("getRememberedAccounts is not supported");
        }
    }

}
