import { Promise, logTelemetryEvent, id } from '@mecontrol/web-inline';
import {
    IAccount, SignInExperienceType, AccountType,
    IV1BackCompatConfig
} from '@mecontrol/public-api';
import {
    IAuthOperations,
    IAuthProvider,
    IAuthNavigationProvider,
    ISignInArgs,
    ISignInToArgs,
    ISignOutFromAppArgs,
    ISignOutFromIdpArgs,
    ISwitchArgs,
    ISwitchToArgs,
    ISignOutAndForgetFromIdpArgs,
    IRPData,
    IDP,
    ErrorSeverity
} from '@mecontrol/common';
import {
    createMsaAccountsProvider,
    createAadAccountsProvider,
    IRememberedAccountsProvider
} from './webAccountProviders';
import { isFunction, setQueryParams, format, isNonEmptyString, hasQueryParam, getHostname } from '../../../utilities';
import {
    createSignInRequest,
    createSignInToRequest,
    createSignOutRequest,
    createSignOutFromIdpRequest,
    createSignOutAndForgetRequest
} from './urlHelpers';
export function createV1BackCompatAuthProvider(config: IV1BackCompatConfig): IAuthProvider {
    return new V1AuthProvider(config);
}

const forgetUserUrlTemplate: string = "https://{0}/forgetuser";

class V1AuthProvider implements IAuthProvider {
    private msaAccountsProvider?: IRememberedAccountsProvider;
    private aadAccountsProvider?: IRememberedAccountsProvider;

    public navProvider: IAuthNavigationProvider;

    private rpData: IRPData;

    //TODO: I really don't know how this is used yet - it seems like it's always Default in all of the providers
    public supportedSignInAccountTypes!: SignInExperienceType;

    constructor(private config: IV1BackCompatConfig) {
        // save away the V1 config data - we'll need this to make decisions on the fly
        this.rpData = config.rpData;

        // Determining the supported sign in experiences is based on the following logic:
        //
        // +---+--------------------------+--------------------------+----------------------------+-----------------------------------------------------+-------------------------------------+
        // | # | rpData.msaInfo is usable | rpData.aadInfo is usable | rpData.aadInfo.blockMsaFed | SignInExperienceType                                | Notes                               |
        // +---+--------------------------+--------------------------+----------------------------+-----------------------------------------------------+-------------------------------------+
        // | 1 | Yes                      | No                       | N/A                        | SignInExperienceType.Msa                            |                                     |
        // | 2 | Yes                      | Yes                      | Yes                        | SignInExperienceType.Msa | SignInExperienceType.Aad |                                     |
        // | 3 | Yes                      | Yes                      | No                         | SignInExperienceType.Converged                      |                                     |
        // | 4 | No                       | No                       | N/A                        | Error - no IDP defined.                             |                                     |
        // | 5 | No                       | Yes                      | Yes                        | SignInExperienceType.Aad                            |                                     |
        // | 6 | No                       | Yes                      | No                         | SignInExperienceType.Converged                      | Get SignInUrl from AAD rpData       |
        // +---+--------------------------+--------------------------+----------------------------+-----------------------------------------------------+-------------------------------------+

        // Why use the ".msaInfo is usable" concept instead of just .msaInfo existence?
        // Why use the ".aadInfo is usable" concept instead of just .aadInfo existence?
        // I'll tell you why, my friend! - because the configs our partners send us are often garbage. There's more than one example of
        // the .msaInfo object existing but the signInUrl not - or being set to an empty string...
        // In fact, even AMC (!) sends an aadInfo object with signOutUrl set to "" and blockMsaFed = true... but doesn't support AAD. :/
        // On the flip side, OFFICE sends an msaInfo object with empty strings for both signInUrl and signOutUrl...

        // this might seem a little winded, but is written this way to be easy to map against the above truth table.
        // It also makes it much easier for us to tweak the response if we decide the assumptions above aren't correct.
        if (isNonEmptyString(this.rpData.msaInfo.signInUrl)) {
            // cases #1,2,3

            if (isNonEmptyString(this.rpData.aadInfo.signInUrl)) {
                // cases #2,3

                // tslint:disable-next-line:prefer-conditional-expression
                if (!!this.rpData.aadInfo.blockMsaFed) {
                    // case #2
                    this.supportedSignInAccountTypes = SignInExperienceType.Converged;
                } else {
                    // case #3
                    this.supportedSignInAccountTypes = SignInExperienceType.Converged;
                }
            } else {
                // case #1
                this.supportedSignInAccountTypes = SignInExperienceType.Msa;
            }
        } else {
            // cases #4,5,6

            if (isNonEmptyString(this.rpData.aadInfo.signInUrl)) {
                // cases #5,6

                // tslint:disable-next-line:prefer-conditional-expression
                if (!!this.rpData.aadInfo.blockMsaFed) {
                    // case #5
                    this.supportedSignInAccountTypes = SignInExperienceType.Aad;
                } else {
                    // case #6
                    this.supportedSignInAccountTypes = SignInExperienceType.Converged;
                }
            } else {
                // case #4
                // If there is neither signInUrls for both aadInfo and msaInfo, attempt to construct a valid signInUrl from location.href
                // Also log a telemetry event to compile a list of affected partners with this issue.
                const signInUrl = format("https://login.microsoftonline.com/common/oauth2/authorize?response_type=id_token&client_id={0}&scope=openid&nonce={1}&response_mode=form_post&redirect_uri={2}", this.rpData.aadInfo.appId as string, id, location.href);
                this.rpData.preferredIdp == IDP.AAD ? this.rpData.aadInfo.signInUrl = signInUrl : this.rpData.msaInfo.signInUrl = signInUrl;
                const warningMessage = `WARNING: Unable to find signInUrl from the rpData object provided. Attempting to construct redirect uri from location.href!`;
                logTelemetryEvent({
                    eventType: "ClientError",
                    name: "Undefined signInUrl",
                    type: "UndefinedAuthState",
                    details: warningMessage,
                    severity: ErrorSeverity.Warning,
                    displayed: false
                });
                // tslint:disable-next-line:no-console
                console.warn(warningMessage);

            }
        }

        // setup the Url provider...
        this.navProvider = new V1BackCompatAuthNavProvider(this.config);

        // Figure out and boot up MSA, AAD or both providers
        // For now, we'll use signInUrl as a marker that we should
        // support a particular IDP. This may not prove to be sufficient
        // but it's all we have for now.
        if (isNonEmptyString(config.rpData.msaInfo.signInUrl) && this.rpData.msaInfo.meUrl) {
            this.msaAccountsProvider = createMsaAccountsProvider(
                {
                    signOutAndForgetUrl: (args: ISignOutAndForgetFromIdpArgs): string => {
                        return this.rpData.msaInfo.signOutUrl;
                    },
                    rememberedAccountsUrl: this.rpData.msaInfo.meUrl as string
                },
            );
        }

        if (isNonEmptyString(config.rpData.aadInfo.signInUrl) && this.rpData.aadInfo.meUrl) {
            this.aadAccountsProvider = createAadAccountsProvider({
                signOutUrl: this.rpData.aadInfo.signOutUrl,
                signOutAndForgetUrl: this.rpData.aadInfo.signOutUrl,
                rememberedAccountsUrl: this.rpData.aadInfo.meUrl as string,
                forgetUrl: format(forgetUserUrlTemplate, getHostname(this.rpData.aadInfo.meUrl as string))
            });
        }
    }

    public isOperationSupported(authOperation: keyof IAuthOperations, accountType: AccountType): boolean {
        if (accountType == AccountType.AAD || accountType == AccountType.MSA_FED) {
            switch (authOperation) {
                case "signIn":
                    return isNonEmptyString(this.rpData.aadInfo.signInUrl);
                case "switch":
                    // ensure expected URL and check for query params that were expected in V1. These checks came from the V1 codebase to keep consitant.
                    return this.rpData.aadInfo.allowNonAadUrls
                    || (hasQueryParam(this.rpData.aadInfo.signInUrl, 'client_id')
                        && hasQueryParam(this.rpData.aadInfo.signInUrl, 'redirect_uri'));
                case "signOutFromApp":
                    return isNonEmptyString(this.rpData.aadInfo.signOutUrl);
                case "signInTo":
                case "switchTo":
                case "signOutFromIdp":
                case "signOutAndForgetFromIdp":
                case "getRememberedAccounts":
                    // call out all actions to ensure we ddin't miss anything. These actions were never supported on V1 for AAD.
                    return false;
                default:
                    return false;
            }
        } else if (accountType == AccountType.MSA) {
            switch (authOperation) {
                case "signIn":
                    return isNonEmptyString(this.rpData.msaInfo.signInUrl);
                case "switch":
                    // ensure expected URL and check for query params that were expected in V1. These checks came from the V1 codebase to keep consitant.
                    return isLoginLiveUrl(this.rpData.msaInfo.signInUrl)
                        && hasQueryParam(this.rpData.msaInfo.signOutUrl, 'ru')
                        && hasQueryParam(this.rpData.msaInfo.signInUrl, 'wreply');
                case "signInTo":
                    return isLoginLiveUrl(this.rpData.msaInfo.signInUrl);
                case "signOutFromApp":
                    return isNonEmptyString(this.rpData.msaInfo.signOutUrl);
                case "signOutAndForgetFromIdp":
                    return isLoginLiveUrl(this.rpData.msaInfo.signOutUrl)
                        && isLoginLiveUrl(this.rpData.msaInfo.meUrl);
                case "getRememberedAccounts":
                    return isLoginLiveUrl(this.rpData.msaInfo.meUrl);
                case "switchTo":
                case "signOutFromIdp":
                    // call out the actions that are NOT supported JUST to ensure we covered everything. These actions were either not supported in V1 or MSA cannot perform them.
                    return false;
                default:
                    return false;
            }
        }

        return false;
    }

    public signIn(args: ISignInArgs): void {
        // Do nothing
    }

    public signInTo(args: ISignInToArgs): void {
        // Do nothing
    }

    public signOutFromApp(args: ISignOutFromAppArgs): void {
        // Do nothing
    }

    public signOutFromIdp(args: ISignOutFromIdpArgs): Promise<void> {
        switch (args.account.type) {
            case AccountType.AAD:
            case AccountType.MSA_FED:
                if (this.aadAccountsProvider) {
                    return this.aadAccountsProvider.signOutFromIdp(args);
                } else {
                    return Promise.reject("SignOut was requested for an account of type AAD or MSA_FED but V1 config didn't provide any usable supporting details in the 'rpData' property for AAD or MSA_FED accounts.");
                }

            case AccountType.MSA:
                if (this.msaAccountsProvider) {
                    return this.msaAccountsProvider.signOutFromIdp(args);
                } else {
                    return Promise.reject("SignOut was requested for an account of type MSA but V1 config didn't provide any usable supporting details in the 'rpData' property for MSA accounts.");
                }
        }
    }

    public signOutAndForgetFromIdp(args: ISignOutAndForgetFromIdpArgs): Promise<void> {
        switch (args.account.type) {
            case AccountType.AAD:
            case AccountType.MSA_FED:
                if (this.aadAccountsProvider) {
                    return this.aadAccountsProvider.signOutAndForgetFromIdp(args);
                } else {
                    return Promise.reject("SignOutAndForgetFromIdp was requested for an account of type AAD or MSA_FED but V1 config didn't provide any usable supporting details in the 'rpData' property for AAD or MSA_FED accounts.");
                }
            case AccountType.MSA:
                if (this.msaAccountsProvider) {
                    return this.msaAccountsProvider.signOutAndForgetFromIdp(args);
                } else {
                    return Promise.reject("SignOutAndForgetFromIdp was requested for an account of type MSA but V1 config didn't provide any usable supporting details in the 'rpData' property for MSA accounts.");
                }
        }
    }

    public switch(args: ISwitchArgs): void {
        // Do nothing
    }

    public switchTo(args: ISwitchToArgs): void {
        // Do nothing
    }

    public getRememberedAccounts(): Promise<IAccount[]> {
        return Promise.all([
            this.msaAccountsProvider ? this.msaAccountsProvider.getRememberedAccounts() : [],
            this.aadAccountsProvider ? this.aadAccountsProvider.getRememberedAccounts() : [],
        ]).then(accountLists => accountLists.reduce((acc, cur) => acc.concat(cur), [])); // Flatten list
    }
}

class V1BackCompatAuthNavProvider implements IAuthNavigationProvider {
    constructor(
        private config: IV1BackCompatConfig
    ) { }

    // V1 notes:
    // -----------------------------
    // signInUrl = appendContextParam(_authStateModel.getSignInUrl(undefined, this.mobileState() === MobileState.Mobile));
    // -----------------------------
    // getSignInUrl(idp?: V1AccountType, isMobile?: boolean): string {
    //     idp = idp ||
    //          (hasActiveUser() && _activeUserData && _activeUserData.idp) ||
    //          (_rpData.preferredIdp === IDP.AAD ? V1AccountType.AAD : V1AccountType.MSA) ||
    //          V1AccountType.MSA;
    //     let isAad: boolean = idp === V1AccountType.AAD || idp === V1AccountType.MSA_FED;
    //     let idpInfo = isAad ? _rpData.aadInfo : _rpData.msaInfo;
    //     let signInUrl = idpInfo.signInUrl;
    //     signInUrl = isAad ? getAadSignInUrl(idpInfo as IAADInfo) : getMsaSignInUrl(idpInfo, isMobile);
    //     return signInUrl;
    // }

    public getSignInUrl(args: ISignInArgs): string | undefined {
        switch (args.signInType) {
            case SignInExperienceType.Msa:
                if (isNonEmptyString(this.config.rpData.msaInfo.signInUrl)) {
                    let url: string = this.config.rpData.msaInfo.signInUrl;
                    if (isFunction(this.config.rpData.msaInfo.generateSignInReturnUrl)) {
                        url = setQueryParams(url, { wreply: this.config.rpData.msaInfo.generateSignInReturnUrl() });
                    }

                    return createSignInRequest(url, args);
                } else {
                    throw new Error("Got signInUrl request for MSA, but we don't have usable MSA info from V1 config.");
                }
            case SignInExperienceType.Aad:
                if (isNonEmptyString(this.config.rpData.aadInfo.signInUrl)) {

                    let url: string = this.config.rpData.aadInfo.signInUrl;
                    if (isFunction(this.config.rpData.aadInfo.generateSignInReturnUrl)) {
                        url = setQueryParams(url, { wreply: this.config.rpData.aadInfo.generateSignInReturnUrl() });
                    }
                    if (this.config.rpData.aadInfo.blockMsaFed) {
                        url = setQueryParams(url, { msafed: '0' });
                    }
                    return createSignInRequest(url, args);
                } else {
                    throw new Error("Got signInUrl request for AAD, but we don't have usable AAD info from V1 config.");
                }
            case SignInExperienceType.Converged:
                if (this.config.rpData.preferredIdp === IDP.AAD && isNonEmptyString(this.config.rpData.aadInfo.signInUrl)) {
                    let url: string = this.config.rpData.aadInfo.signInUrl;
                    if (isFunction(this.config.rpData.aadInfo.generateSignInReturnUrl)) {
                        url = setQueryParams(url, { wreply: this.config.rpData.aadInfo.generateSignInReturnUrl() });
                    }
                    if (this.config.rpData.aadInfo.blockMsaFed) {
                        url = setQueryParams(url, { msafed: '0' });
                    }
                    return createSignInRequest(url, args);
                } else if (isNonEmptyString(this.config.rpData.msaInfo.signInUrl)) {
                    // in a pinch, try to use the MSA endpoint (yes, there is terrible config data that does this, and V1 does gymnastics)
                    let url: string = this.config.rpData.msaInfo.signInUrl;
                    if (isFunction(this.config.rpData.msaInfo.generateSignInReturnUrl)) {
                        url = setQueryParams(url, { wreply: this.config.rpData.msaInfo.generateSignInReturnUrl() });
                    }

                    return createSignInRequest(url, args);
                } else {
                    throw new Error("Got signInUrl request for converged experience, but we don't have usable AAD info (or MSA info either) from V1 config.");
                }
        }
    }

    public getSignInToUrl(args: ISignInToArgs): string | undefined {
        switch (args.nextAccount.type) {
            case AccountType.MSA:
                if (isNonEmptyString(this.config.rpData.msaInfo.signInUrl)) {
                    let url: string = this.config.rpData.msaInfo.signInUrl;
                    if (isFunction(this.config.rpData.msaInfo.generateSignInReturnUrl)) {
                        url = setQueryParams(url, { wreply: this.config.rpData.msaInfo.generateSignInReturnUrl() });
                    }
                    if (isLoginLiveUrl(url)) {
                        return setQueryParams(url, { username: args.nextAccount.memberName });
                    }
                    return createSignInToRequest(url, args);
                } else {
                    throw new Error("Got signInToUrl request for MSA, but we don't have usable MSA info from V1 config.");
                }
            case AccountType.AAD:
            case AccountType.MSA_FED:
                if (isNonEmptyString(this.config.rpData.aadInfo.signInUrl)) {

                    let url: string = this.config.rpData.aadInfo.signInUrl;
                    if (isFunction(this.config.rpData.aadInfo.generateSignInReturnUrl)) {
                        url = setQueryParams(url, { wreply: this.config.rpData.aadInfo.generateSignInReturnUrl() });
                    }
                    if (this.config.rpData.aadInfo.blockMsaFed) {
                        url = setQueryParams(url, { msafed: '0' });
                    }
                    return createSignInToRequest(url, args);
                } else {
                    throw new Error("Got signInToUrl request for AAD, but we don't have usable AAD info from V1 config.");
                }
        }
    }

    public getSignOutFromAppUrl(args: ISignOutFromAppArgs): string | undefined {
        switch (args.currentAccount.type) {
            case AccountType.MSA:
                if (isNonEmptyString(this.config.rpData.msaInfo.signOutUrl)) {
                    let url: string = this.config.rpData.msaInfo.signOutUrl;
                    if (isFunction(this.config.rpData.msaInfo.generateSignOutReturnUrl)) {
                        url = setQueryParams(url, { wreply: this.config.rpData.msaInfo.generateSignOutReturnUrl() });
                    }
                    return createSignOutRequest(url, args);
                } else {
                    throw new Error("Got signOutFromAppUrl request for MSA, but we don't have usable MSA info from V1 config.");
                }
            case AccountType.AAD:
            case AccountType.MSA_FED:
                if (isNonEmptyString(this.config.rpData.aadInfo.signOutUrl)) {

                    let url: string = this.config.rpData.aadInfo.signOutUrl;
                    if (isFunction(this.config.rpData.aadInfo.generateSignOutReturnUrl)) {
                        url = setQueryParams(url, { wreply: this.config.rpData.aadInfo.generateSignOutReturnUrl() });
                    }
                    return createSignOutRequest(url, args);
                } else {
                    throw new Error("Got signOutFromAppUrl request for AAD, but we don't have usable AAD info from V1 config.");
                }
        }
    }

    public getSignOutFromIdpUrl(args: ISignOutFromIdpArgs): string | undefined {
        switch (args.account.type) {
            case AccountType.MSA:
            case AccountType.MSA_FED:
                throw new Error("Method not supported for MSA and MSA_FED Accounts.");
            case AccountType.AAD:
                if (isNonEmptyString(this.config.rpData.aadInfo.signOutUrl)) {
                    return createSignOutFromIdpRequest(this.config.rpData.aadInfo.signOutUrl, args);
                } else {
                    throw new Error("Got signOutFromAppUrl request for AAD, but we don't have usable AAD info from V1 config.");
                }
        }
    }

    public getSignOutAndForgetFromIdpUrl(args: ISignOutAndForgetFromIdpArgs): string | undefined {
        switch (args.account.type) {
            case AccountType.MSA:
                if (isNonEmptyString(this.config.rpData.msaInfo.signOutUrl)) {
                    return createSignOutAndForgetRequest(this.config.rpData.msaInfo.signOutUrl, args);
                } else {
                    throw new Error("Got signOutFromAppUrl request for MSA, but we don't have usable AAD info from V1 config.");
                }
            case AccountType.AAD:
            case AccountType.MSA_FED:
                if (isNonEmptyString(this.config.rpData.aadInfo.signOutUrl)) {
                    return createSignOutFromIdpRequest(this.config.rpData.aadInfo.signOutUrl, args);
                } else {
                    throw new Error("Got signOutAndForgetFromAppUrl request for AAD, but we don't have usable AAD info from V1 config.");
                }
        }
    }

    public getSwitchUrl(args: ISwitchArgs): string | undefined {
        return this.getSwitchOrSwitchToUrl(args, true);
    }

    public getSwitchToUrl(args: ISwitchToArgs): string | undefined {
        return this.getSwitchOrSwitchToUrl(args, false);
    }

    public getSwitchOrSwitchToUrl(args: ISwitchArgs | ISwitchToArgs, isJustSwitch: boolean): string | undefined {
        let signInType: SignInExperienceType = (args as ISwitchArgs).signInType || this.config.rpData.preferredIdp === IDP.MSA ? SignInExperienceType.Msa : SignInExperienceType.Aad;

        // Get the sign in to URL...
        let signInUrl: string | undefined =
            isJustSwitch ?
                this.getSignInUrl({ signInType }) :
                this.getSignInToUrl({ nextAccount: (args as ISwitchToArgs).nextAccount });

        // if we're currently in an AAD account, then we simply go to the sign in url...
        if (args.currentAccount.type === AccountType.AAD) {
            return setQueryParams(signInUrl!, { prompt: 'select_account' });
        } else {
            // so, we know that we're currently signed in as an MSA (or MSA_FED)
            // IF the SignInUrl goes to "login.live.com", then we need to sign out first...
            if (isLoginLiveUrl(signInUrl)) {
                let signOutUrl: string | undefined = this.getSignOutFromAppUrl({ currentAccount: args.currentAccount } as ISignOutFromAppArgs);

                // try to figure out where we're headed since the way forwarding works is different per IDP...
                if (signOutUrl) {
                    // tslint:disable-next-line:prefer-conditional-expression
                    if (this.config.rpData.preferredIdp === IDP.AAD) {
                        // try to get to AAD after this...
                        return setQueryParams(signOutUrl, { wreply: signInUrl });
                    } else {
                        // try to get to MSA after this...
                        return setQueryParams(signOutUrl, { ru: signInUrl, lru: signInUrl });
                    }
                } else {
                    // no good answer to "what if no signOutUrl", but we should at least try to get to the sign in, I guess...
                    return signInUrl;
                }
            } else {
                // just return the signInUrl as it appears that the partner is trying to handle this themselves...
                return signInUrl;
            }
        }
    }
}

/**
 * Tests if the given URL is valid and is hosted on login.live.com.
 * @param url The url string to validate.
 */
function isLoginLiveUrl(url: string | undefined): boolean {
    if (!url) {
        return false;
    }

    return getHostname(url).toLowerCase().indexOf('login.live.com') > -1;
}
