import { IWebAadWithMsaFedConfig, SignInExperienceType, AccountType, IAccount } from '@mecontrol/public-api';
import {
    IAuthOperations,
    IAuthProvider,
    IAuthNavigationProvider,
    ISignInArgs,
    ISignInToArgs,
    ISignOutFromAppArgs,
    ISignOutFromIdpArgs,
    ISignOutAndForgetFromIdpArgs,
    ISwitchArgs,
    ISwitchToArgs,
} from '@mecontrol/common';
import { Promise, ME, assertNever, createError } from '@mecontrol/web-inline';
import { IRememberedAccountsProvider, createAadAccountsProvider } from './webAccountProviders';
import { isStringOrFunction } from '../../../utilities';
import { canLeverageRememberedAccounts } from './operationHelpers';
import { WebAuthNavProvider } from './navProviders';

export function createWebAadWithMsaFedAuthProvider(config: IWebAadWithMsaFedConfig): IAuthProvider {
    return new WebAadWithMsaFedProvider(config);
}

class WebAadWithMsaFedProvider implements IAuthProvider {
    private webAadIdpProvider: IRememberedAccountsProvider;

    public navProvider: IAuthNavigationProvider;
    public supportedSignInAccountTypes: SignInExperienceType = SignInExperienceType.Aad;

    constructor(private config: IWebAadWithMsaFedConfig) {
        this.navProvider = new WebAuthNavProvider(this.config);
        this.webAadIdpProvider = createAadAccountsProvider(this.config.aad);
    }

    public isOperationSupported(authOperation: keyof IAuthOperations, accountType: AccountType): boolean {
        if (accountType !== AccountType.AAD && accountType !== AccountType.MSA_FED) {
            return false;
        }

        switch (authOperation) {
            case "signIn":
                return isStringOrFunction(this.config.appSignInUrl);
            case "signInTo":
                return isStringOrFunction(this.config.appSignInToUrl);
            case "switch":
                return isStringOrFunction(this.config.appSwitchUrl);
            case "switchTo":
                return isStringOrFunction(this.config.appSwitchToUrl);
            case "signOutFromApp":
                return isStringOrFunction(this.config.appSignOutUrl);
            case "signOutFromIdp":
                return isStringOrFunction(this.config.aad?.signOutUrl);
            case "signOutAndForgetFromIdp":
                return isStringOrFunction(this.config.aad?.signOutAndForgetUrl);
            case "getRememberedAccounts":
                return ME.Config.aad
                    && ME.Config.remAcc
                    && canLeverageRememberedAccounts(this, accountType)
                    && isStringOrFunction(this.config.aad?.rememberedAccountsUrl);
        }

        assertNever(authOperation, `Operation "${authOperation}" was unhandled.`);
    }

    public signIn(args: ISignInArgs): void {
        // Do nothing
    }

    public signInTo(args: ISignInToArgs): void {
        // Do nothing
    }

    public signOutFromApp(args: ISignOutFromAppArgs): void {
        // Do nothing
    }

    public signOutFromIdp(args: ISignOutFromIdpArgs): Promise<void> {
        if (this.isOperationSupported('signOutFromIdp', args.account.type)) {
            return this.webAadIdpProvider.signOutFromIdp(args);
        }

        return Promise.reject(createError('signOutFromIdp is not supported'));
    }

    public signOutAndForgetFromIdp(args: ISignOutAndForgetFromIdpArgs): Promise<void> {
        if (this.isOperationSupported('signOutAndForgetFromIdp', args.account.type)) {
            return this.webAadIdpProvider.signOutAndForgetFromIdp(args);
        }

        return Promise.reject(createError('signOutAndForgetFromIdp is not supported'));
    }

    public switch(args: ISwitchArgs): void {
        // Do nothing
    }

    public switchTo(args: ISwitchToArgs): void {
        // Do nothing
    }

    public getRememberedAccounts(): Promise<IAccount[]> {
        if (this.isOperationSupported('getRememberedAccounts', AccountType.AAD)) {
            return this.webAadIdpProvider.getRememberedAccounts();
        }

        return Promise.reject(createError('getRememberedAccounts is not supported'));
    }
}
