import {
    IAccount,
    SignInExperienceType,
    AccountType,
    IWebAadWithMsaProxyConfig
} from '@mecontrol/public-api';
import {
    IAuthOperations,
    IAuthProvider,
    IAuthNavigationProvider,
    ISignInArgs,
    ISignInToArgs,
    ISignOutFromAppArgs,
    ISignOutFromIdpArgs,
    ISwitchArgs,
    ISwitchToArgs,
    ISignOutAndForgetFromIdpArgs
} from '@mecontrol/common';
import {
    Promise,
    ME,
    assertNever,
    logTelemetryEvent,
    toJsonable,
    createError
} from '@mecontrol/web-inline';
import {
    createMsaAccountsProvider,
    createAadAccountsProvider,
    IRememberedAccountsProvider
} from './webAccountProviders';
import { isStringOrFunction } from '../../../utilities';
import { canLeverageRememberedAccounts } from './operationHelpers';
import { WebAuthNavProvider } from './navProviders';

export function createWebAadWithMsaProxyProvider(
    config: IWebAadWithMsaProxyConfig
): IAuthProvider {
    return new WebAadWithMsaProxyProvider(config);
}

class WebAadWithMsaProxyProvider implements IAuthProvider {
    private msaAccountsProvider: IRememberedAccountsProvider;
    private aadAccountsProvider: IRememberedAccountsProvider;

    public navProvider: IAuthNavigationProvider;

    //TODO: I really don't know how this is used yet - it seems like it's always Default in all of the providers
    public supportedSignInAccountTypes: SignInExperienceType =
        SignInExperienceType.Converged;

    constructor(private config: IWebAadWithMsaProxyConfig) {
        this.navProvider = new WebAuthNavProvider(this.config);
        this.aadAccountsProvider = createAadAccountsProvider(this.config.aad, {
            proxyToOtherIdp: true
        });
        this.msaAccountsProvider = createMsaAccountsProvider(this.config.msa);
    }

    public isOperationSupported(
        authOperation: keyof IAuthOperations,
        accountType: AccountType
    ): boolean {
        switch (authOperation) {
            case 'signIn':
                return isStringOrFunction(this.config.appSignInUrl);
            case 'signInTo':
                return isStringOrFunction(this.config.appSignInToUrl);
            case 'signOutFromApp':
                return isStringOrFunction(this.config.appSignOutUrl);
            case 'signOutFromIdp':
                return (
                    (accountType === AccountType.AAD ||
                        accountType === AccountType.MSA_FED) &&
                    isStringOrFunction(this.config.aad?.signOutUrl)
                );
            case 'signOutAndForgetFromIdp':
                return (
                    ((accountType === AccountType.AAD || accountType === AccountType.MSA_FED) &&
                        isStringOrFunction(
                            this.config.aad?.signOutAndForgetUrl
                        )) ||
                    (accountType === AccountType.MSA &&
                        isStringOrFunction(
                            this.config.msa?.signOutAndForgetUrl
                        ))
                );
            case 'switch':
                return isStringOrFunction(this.config.appSwitchUrl);
            case 'switchTo':
                return isStringOrFunction(this.config.appSwitchToUrl);
            case 'getRememberedAccounts':
                return (
                    ME.Config.remAcc &&
                    canLeverageRememberedAccounts(this, accountType) &&
                    isStringOrFunction(this.config.aad?.rememberedAccountsUrl)
                );
        }

        assertNever(
            authOperation,
            `Operation "${authOperation}" was unhandled.`
        );
    }

    public signIn(args: ISignInArgs): void {
        // Do nothing
    }

    public signInTo(args: ISignInToArgs): void {
        // Do nothing
    }

    public signOutFromApp(args: ISignOutFromAppArgs): void {
        // Do nothing
    }

    public signOutFromIdp(args: ISignOutFromIdpArgs): Promise<void> {
        const acctType = args.account.type;
        if (this.isOperationSupported('signOutFromIdp', acctType)) {
            switch (acctType) {
                case AccountType.AAD:
                case AccountType.MSA_FED:
                    return this.aadAccountsProvider.signOutFromIdp(args);
                case AccountType.MSA:
                    // The MSA Account provider in this scenario isn't configured with rememberedAccounts
                    // so we need the remembered accounts call to go through this authProvider so we
                    // pass in the authProvider as a parameter
                    return this.msaAccountsProvider.signOutFromIdp(args, this);
            }
        }

        return Promise.reject(createError('signOutFromIdp is not supported'));
    }

    public signOutAndForgetFromIdp(
        args: ISignOutAndForgetFromIdpArgs
    ): Promise<void> {
        const acctType = args.account.type;
        if (this.isOperationSupported('signOutAndForgetFromIdp', acctType)) {
            switch (acctType) {
                case AccountType.AAD:
                case AccountType.MSA_FED:
                    return this.aadAccountsProvider.signOutAndForgetFromIdp(args);
                case AccountType.MSA:
                    // The MSA Account provider in this scenario isn't configured with rememberedAccounts
                    // so we need the remembered accounts call to go through this authProvider so we
                    // pass in the authProvider as a parameter
                    return this.msaAccountsProvider.signOutAndForgetFromIdp(args, this);
            }
        }

        return Promise.reject(
            createError('signOutAndForgetFromIdp is not supported')
        );
    }

    public switch(args: ISwitchArgs): void {
        // Do nothing
    }

    public switchTo(args: ISwitchToArgs): void {
        // Do nothing
    }

    public getRememberedAccounts(): Promise<IAccount[]> {
        const handleFailure = (type: AccountType) => (error: Error) => {
            logTelemetryEvent({
                eventType: 'ClientError',
                name: 'Failed to retrieve remembered accounts for ' + type,
                type: 'RememberedAccountsFailure',
                details: JSON.stringify(toJsonable(error)),
                displayed: false
            });

            let empty: IAccount[] = [];
            return empty;
        };

        const msaSupported = this.isOperationSupported(
            'getRememberedAccounts',
            AccountType.MSA
        );
        const aadSupported = this.isOperationSupported(
            'getRememberedAccounts',
            AccountType.AAD
        );

        if (!msaSupported && !aadSupported) {
            return Promise.reject(
                createError('getRememberedAccounts is not supported')
            );
        }

        return this.aadAccountsProvider
            .getRememberedAccounts()
            .catch(handleFailure(AccountType.AAD))
            .then(accList =>
                accList.filter(acc => acc.type != AccountType.MSA_FED)
            );
    }
}
