import { IAccount, IWebIdpConfig, IMsaWebIdpConfig, IAadWebIdpConfig, AccountType, AuthenticatedState, AuthProviderConfigType } from '@mecontrol/public-api';
import { createError, ME, Promise, id, logTelemetryEvent } from '@mecontrol/web-inline';
import { ISignOutFromIdpArgs, ISignOutAndForgetFromIdpArgs, RememberedAccountErrors, IAuthProvider } from '@mecontrol/common';
import { isString, setQueryParams, findAccount } from '../../../utilities';
import { openIframe, IIFrameOperation } from '../../iframe';
import { userDataToIAccount } from '../selectors';
import { IRememberedAccountsResponse } from '../userData';
import { createSignOutAndForgetRequest, createSignOutFromIdpRequest, createForgetRequest } from './urlHelpers';

export interface IRememberedAccountsProvider {
    signOutFromIdp(args: ISignOutFromIdpArgs, authProvider?: IAuthProvider): Promise<void>;
    signOutAndForgetFromIdp(args: ISignOutAndForgetFromIdpArgs, authProvider?: IAuthProvider): Promise<void>;
    getRememberedAccounts(): Promise<IAccount[]>;
}

// TODO: Move this to a more appropriate module
export interface ISignedOutResponse {
    signoutStatus: string;
    error?: string;
}

export interface IWebRememberedAccountsProviderOptions {
    proxyToOtherIdp: boolean;
}

const defaultOptions: IWebRememberedAccountsProviderOptions = {
    proxyToOtherIdp: false,
};

export function createMsaAccountsProvider(
    config: Partial<IMsaWebIdpConfig>,
    options: IWebRememberedAccountsProviderOptions = { ...defaultOptions }
): IRememberedAccountsProvider {
    return new MsaWebAccountsProvider(config, {
        proxyToOtherIdp: options.proxyToOtherIdp && ME.Config.aad && ME.Config.pxy
    });
}

export function createAadAccountsProvider(
    config: Partial<IAadWebIdpConfig>,
    options: IWebRememberedAccountsProviderOptions = { ...defaultOptions }
): IRememberedAccountsProvider {
    return new AadWebAccountsProvider(config, {
        proxyToOtherIdp: options.proxyToOtherIdp && ME.Config.pxy
    });
}

class AadWebAccountsProvider implements IRememberedAccountsProvider {

    private idpName: string = 'AAD';

    constructor(
        private config: Partial<IAadWebIdpConfig>,
        private options: IWebRememberedAccountsProviderOptions
    ) { }

    public signOutFromIdp(args: ISignOutFromIdpArgs, authProvider?: IAuthProvider): Promise<void> {
        const operation = getSignOutFromIdpOp(this.idpName);
        const iframeUrl = this.getSignOutIframeUrl(args);
        return this.openSignOutIframe(
            iframeUrl,
            args.account,
            false,
            operation,
            authProvider
        );
    }

    public signOutAndForgetFromIdp(args: ISignOutAndForgetFromIdpArgs, authProvider?: IAuthProvider): Promise<void> {
        const accountAuthState = args.account.authenticatedState;
        if (accountAuthState === AuthenticatedState.NotSignedIn) {
            const operation = getForgetFromIdpOp(this.idpName);
            const iframeUrl = this.getForgetIframeUrl(args);
            return openIframe(iframeUrl, operation, 0)
                .then(() => authProvider ? authProvider.getRememberedAccounts() : this.getRememberedAccounts())
                .then(accounts => {
                    if (findAccount(accounts, args.account) != undefined) {
                        throw createError('Forget user failed. IDP getRememberedAccounts contains account user attempted to forget.');
                    }
                });
        }
        else {
            const operation = getSignOutAndForgetFromIdpOp(this.idpName);
            const iframeUrl = this.getSignOutAndForgetIframeUrl(args);
            return this.openSignOutIframe(iframeUrl, args.account, true, operation, authProvider);

        }
    }

    public getRememberedAccounts(): Promise<IAccount[]> {
        if (!ME.Config.aad) {
            return Promise.reject(createError('AAD Remembered Accounts disabled'));
        }

        const operation = getRememberedAccountsOp(this.idpName);
        return getRememberedAccounts(this.config.rememberedAccountsUrl, this.options, operation);
    }

    private getSignOutIframeUrl(args: ISignOutFromIdpArgs): string {
        if (!this.config.signOutUrl) {
            throw createError('AadWebIdp is not configured for signOutFromIdp.');
        }

        switch (args.account.type) {
            case AccountType.AAD:
            case AccountType.MSA_FED:
                return isString(this.config.signOutUrl)
                    ? createSignOutFromIdpRequest(this.config.signOutUrl, args)
                    : this.config.signOutUrl(args);
            default:
                throw createError(`AadWebAccountsProvider: SignOut operation not supported for ${args.account.type}`);
        }
    }

    private getSignOutAndForgetIframeUrl(args: ISignOutAndForgetFromIdpArgs): string {
        if (!this.config.signOutAndForgetUrl) {
            throw createError('AadWebIdp is not configured for signOutAndForgetFromIdp.');
        }

        switch (args.account.type) {
            case AccountType.MSA_FED:
            case AccountType.AAD:
                return isString(this.config.signOutAndForgetUrl)
                    ? createSignOutAndForgetRequest(this.config.signOutAndForgetUrl, args)
                    : this.config.signOutAndForgetUrl(args);
            default:
                throw createError(`AadWebAccountsProvider: SignOutAndForget operation not supported for ${args.account.type}`);
        }
    }

    private getForgetIframeUrl(args: ISignOutAndForgetFromIdpArgs): string {
        if (!this.config.forgetUrl) {
            throw createError('AadWebIdp is not configured for forgetFromIdp');
        }

        switch (args.account.type) {
            case AccountType.MSA_FED:
            case AccountType.AAD:
                return isString(this.config.forgetUrl)
                    ? createForgetRequest(this.config.forgetUrl, args)
                    : this.config.forgetUrl(args);
            default:
                throw createError(`AadWebAccountsProvider: ForgetFromIdp operation not supported for ${args.account.type}`);
        }
    }

    private openSignOutIframe(
        url: string,
        account: IAccount,
        shouldForget: boolean,
        operation: IIFrameOperation,
        authProvider?: IAuthProvider
    ): Promise<void> {
        return openIframe(url, operation, 1)
            .then(processSignedOutMessage)
            .then(() => authProvider ? authProvider.getRememberedAccounts() : this.getRememberedAccounts())
            .then(accounts => {
                const signedOutAccount = findAccount(accounts, account);
                if (shouldForget && signedOutAccount != undefined) {
                    throw createError('Sign out failed. IDP getRememberedAccounts contains account user attempted to remove.');
                }
                else if (signedOutAccount && signedOutAccount.authenticatedState !== AuthenticatedState.NotSignedIn) {
                    throw createError('Sign out failed. IDP did not indicate the account was signed out successfully.');
                }
            });
    }
}

class MsaWebAccountsProvider implements IRememberedAccountsProvider {

    private idpName: string = 'MSA';

    constructor(
        private config: Partial<IMsaWebIdpConfig>,
        private options: IWebRememberedAccountsProviderOptions
    ) { }

    public signOutFromIdp(args: ISignOutFromIdpArgs): Promise<void> {
        return Promise.reject(createError('MsaWebIdp does not support signOutFromIdp.'));
    }

    public signOutAndForgetFromIdp(args: ISignOutAndForgetFromIdpArgs, authProvider?: IAuthProvider): Promise<void> {
        const operation = getSignOutAndForgetFromIdpOp(this.idpName);
        const iframeUrl = this.getSignOutAndForgetIframeUrl(args);
        return this.openSignOutIframe(iframeUrl, args.account, operation, authProvider);
    }

    public getRememberedAccounts(): Promise<IAccount[]> {
        // TODO: for aad, the check is (!ME.Config.aad) but there's no parallel .msa property - is this sufficient?
        // should AAD also check the this.config.rememberedAccountsUrl too?
        if (!this.config.rememberedAccountsUrl) {
            return Promise.reject(createError('MSA Remembered Accounts disabled'));
        }

        const operation = getRememberedAccountsOp(this.idpName);
        return getRememberedAccounts(this.config.rememberedAccountsUrl, this.options, operation);
    }

    private getSignOutAndForgetIframeUrl(args: ISignOutAndForgetFromIdpArgs): string {
        if (!this.config.signOutAndForgetUrl) {
            throw createError('Partner failed to provide a URL or callback to generate a URL for SignOutAndForget.');
        }

        switch (args.account.type) {
            case AccountType.AAD:
            case AccountType.MSA_FED:
                throw createError(`MsaWebAccountsProvider: SignOutAndForget operation not supported for ${args.account.type}`);
            case AccountType.MSA:
                return isString(this.config.signOutAndForgetUrl)
                    ? createSignOutAndForgetRequest(this.config.signOutAndForgetUrl, args)
                    : this.config.signOutAndForgetUrl(args);
        }
    }

    private openSignOutIframe(
        url: string,
        account: IAccount,
        operation: IIFrameOperation,
        authProvider?: IAuthProvider
    ): Promise<void> {
        if (authProvider && (authProvider as any).config.type === AuthProviderConfigType.WebAadWithMsaProxy) {
            return openIframe(url, operation, 1, () => true, authProvider)
                .then(processSignedOutMessage)
                .then(() => authProvider ? authProvider.getRememberedAccounts() : this.getRememberedAccounts())
                .then(accounts => {
                    if (findAccount(accounts, account) != undefined) {
                        throw createError('Sign out failed. IDP getRememberedAccounts contains account user attempted to remove.');
                    }
                })
                .catch(() => {
                    // if signOutIframe fails, attempt to call rememberedAccounts again to see if it was a false failure due to proxy authprovider behavior.
                    const rememberedAccounts = authProvider ? authProvider.getRememberedAccounts() : this.getRememberedAccounts()
                    rememberedAccounts.then(accounts => {
                        if (findAccount(accounts, account) != undefined) {
                            throw createError('Sign out failed. IDP getRememberedAccounts contains account user attempted to remove.');
                        }
                    });
                });
        } else {
            return openIframe(url, operation, 1)
                .then(processSignedOutMessage)
                .then(() => authProvider ? authProvider.getRememberedAccounts() : this.getRememberedAccounts())
                .then(accounts => {
                    if (findAccount(accounts, account) != undefined) {
                        throw createError('Sign out failed. IDP getRememberedAccounts contains account user attempted to remove.');
                    }
                });
        }
    }
}

function isRememberedAccountsMessage(data: any): boolean {
    if (typeof data !== 'string') {
        return false;
    }

    let msg: unknown;
    try {
        msg = JSON.parse(data);
    } catch (e) {
        return false;
    }

    return (
        typeof msg == 'object' &&
        msg != null &&
        ('error' in msg || 'userList' in msg)
    );
}

function getRememberedAccounts(
    rememberedAccountsUrl: IWebIdpConfig["rememberedAccountsUrl"],
    options: IWebRememberedAccountsProviderOptions,
    operation: IIFrameOperation
): Promise<IAccount[]> {
    if (!ME.Config.remAcc) {
        return Promise.reject(createError('Remembered Accounts disabled'));
    }

    if (!rememberedAccountsUrl) {
        return Promise.reject(createError(operation.service + 'WebIdp not configured for getRememberedAccounts.'));
    }

    let iframeUrl: string;
    if (typeof rememberedAccountsUrl === 'function') {
        iframeUrl = rememberedAccountsUrl();
    } else {
        iframeUrl = rememberedAccountsUrl;
    }

    iframeUrl = setQueryParams(iframeUrl, { uaid: id, partnerId: ME.Config.ptn });
    if (options.proxyToOtherIdp) {
        iframeUrl = setQueryParams(iframeUrl, { idpflag: 'proxy' });
    }

    let expectedMessageCount = 1;
    if (options.proxyToOtherIdp) {
        expectedMessageCount = 2;
    }

    // TODO: Should we fire the outgoing request here instead?

    return openIframe(iframeUrl, operation, expectedMessageCount, isRememberedAccountsMessage)
        .then(messages => messages
            .map(processRememberedAccountsMessage)     // convert each message into an IAccount[]
            .reduce((acc, cur) => acc.concat(cur), []) // Flatten array
        );
}

function processRememberedAccountsMessage(message: any): IAccount[] {
    // let dataPoints: any = {
    //     currentIdp: isMsa ? IDP.MSA : IDP.AAD,
    //     messagesMissing: _userStateCall.expectedMessagesCount
    // };

    if (typeof message !== 'string') {
        throw createError('Invalid message from IDP. Was not a string.');
    }

    const accountData: IRememberedAccountsResponse | null = JSON.parse(message);
    if (accountData && accountData.error) {
        if (accountData.error === RememberedAccountErrors.NoReturnedMsaAccountsError) {
            // MSA Returns this error message when no remembered accounts are returned.
            return [];
        }
        else if (accountData.error === RememberedAccountErrors.InvalidProxiedDomain) {
            logTelemetryEvent({
                eventType: 'ClientError',
                name: 'Invalid proxy domain used for get remembered accounts iFrame',
                type: 'RememberedAccountsProxyFailure',
                details: `iFrame hostname: ${window.location.hostname}`,
                displayed: false
            });

            throw createError('Invalid proxy domain used for get remembered accounts iFrame');
        }
        else {
            throw createError(`Remembered Accounts API returned an error: ${accountData.error}`);
        }
    }
    else if (accountData && accountData.userList) {
        // V1 logging: logQos('UserState', success, perfNow() - _userStateCall.startTime, errorCode, dataPoints);
        return accountData.userList.map(userDataToIAccount);
    }
    else {
        // Evaluate V1 error handling

        // let errorCode = 'InvalidIdpData';
        // if (userData && userData.error) {
        //     dataPoints.IDPError = userData.error;
        //     errorCode = 'IDPError';
        // }
        // logQos('UserStatePartial', false, perfNow() - _userStateCall.startTime, errorCode, dataPoints);

        throw createError('MSA Remembered Accounts API returned invalid data');
    }
}

function processSignedOutMessage(messages: any[]): void {
    let message = messages[0];
    if (typeof message !== 'string') {
        throw createError('Invalid message from IDP. Was not a string.');
    }

    if (message === 'signedout') {
        // login.live.com/logout just sends the message "signedout" when the
        // sign out was successful
        return;
    }

    let signedOutMessage: ISignedOutResponse = JSON.parse(message);
    // EvoSts is hardcoded to always return an error here, so we ignore it
    // if (signedOutMessage.error) {
    //     throw createError(`Sign Out operation failed with error: ${signedOutMessage.error.toString()}`);
    // }

    if (signedOutMessage.error && signedOutMessage.error === RememberedAccountErrors.InvalidProxiedDomain) {
        logTelemetryEvent({
            eventType: 'ClientError',
            name: 'Invalid proxy domain used for sign out of remembered account iFrame',
            type: 'RememberedAccountsProxyFailure',
            details: `iFrame hostname: ${window.location.hostname}`,
            displayed: false
        });

        throw createError('Sign out operation failed due to Invalid proxy domain used for sign out of remembered account iFrame');
    }

    let signedoutStatus = signedOutMessage.signoutStatus;
    if (signedoutStatus.toString() !== 'true') {
        throw createError(`Sign out operation failed with signoutStatus: ${signedoutStatus.toString()}`);
    }

    return;
}

type IFrameOperationFactory = (idp: string) => IIFrameOperation;

const getRememberedAccountsOp: IFrameOperationFactory = idp => ({
    name: 'GetRememberedAccounts',
    operation: 'GetRememberedAccounts',
    service: idp.toUpperCase()
});

const getSignOutFromIdpOp: IFrameOperationFactory = idp => ({
    name: 'SignOutFromIdp',
    operation: 'SignOut',
    service: idp.toUpperCase()
});

const getSignOutAndForgetFromIdpOp: IFrameOperationFactory = idp => ({
    name: 'SignOutAndForgetFromIdp',
    operation: 'SignOut',
    service: idp.toUpperCase()
});

const getForgetFromIdpOp: IFrameOperationFactory = idp => ({
    name: 'ForgetFromIdp',
    operation: 'ForgetAccount',
    service: idp.toUpperCase()
});
