import { IAccount, IWebDualStackConfig, SignInExperienceType, AccountType } from '@mecontrol/public-api';
import {
    IAuthOperations,
    IAuthProvider,
    IAuthNavigationProvider,
    ISignInArgs,
    ISignInToArgs,
    ISignOutFromAppArgs,
    ISignOutFromIdpArgs,
    ISwitchArgs,
    ISwitchToArgs,
    ISignOutAndForgetFromIdpArgs
} from '@mecontrol/common';
import { Promise, ME, assertNever, logTelemetryEvent, toJsonable, createError } from '@mecontrol/web-inline';
import { createMsaAccountsProvider, createAadAccountsProvider, IRememberedAccountsProvider } from './webAccountProviders';
import { isStringOrFunction } from '../../../utilities';
import { canLeverageRememberedAccounts } from './operationHelpers';
import { WebAuthNavProvider } from './navProviders';

export function createWebDualStackAuthProvider(config: IWebDualStackConfig): IAuthProvider {
    return new WebDualStackProvider(config);
}

class WebDualStackProvider implements IAuthProvider {
    private msaAccountsProvider: IRememberedAccountsProvider;
    private aadAccountsProvider: IRememberedAccountsProvider;

    public navProvider: IAuthNavigationProvider;

    //TODO: I really don't know how this is used yet - it seems like it's always Default in all of the providers
    public supportedSignInAccountTypes: SignInExperienceType = SignInExperienceType.Converged;

    constructor(private config: IWebDualStackConfig) {
        this.navProvider = new WebAuthNavProvider(this.config);
        this.msaAccountsProvider = createMsaAccountsProvider(this.config.msa);
        this.aadAccountsProvider = createAadAccountsProvider(this.config.aad);
    }

    public isOperationSupported(authOperation: keyof IAuthOperations, accountType: AccountType): boolean {
        switch (authOperation) {
            case "signIn":
                return isStringOrFunction(this.config.appSignInUrl);
            case "signInTo":
                return isStringOrFunction(this.config.appSignInToUrl);
            case "signOutFromApp":
                return isStringOrFunction(this.config.appSignOutUrl);
            case "signOutFromIdp":
                return (
                    (accountType === AccountType.AAD ||
                        accountType === AccountType.MSA_FED) &&
                    isStringOrFunction(this.config.aad?.signOutUrl)
                );
            case "signOutAndForgetFromIdp":
                return (
                    ((accountType === AccountType.AAD || accountType == AccountType.MSA_FED) &&
                        isStringOrFunction(
                            this.config.aad?.signOutAndForgetUrl
                        )) ||
                    (accountType === AccountType.MSA &&
                        isStringOrFunction(
                            this.config.msa?.signOutAndForgetUrl
                        ))
                );
            case "switch":
                return isStringOrFunction(this.config.appSwitchUrl);
            case "switchTo":
                return isStringOrFunction(this.config.appSwitchToUrl);
            case "getRememberedAccounts":
                return ME.Config.remAcc &&
                    canLeverageRememberedAccounts(this, accountType) &&
                    (
                        (accountType === AccountType.AAD && isStringOrFunction(this.config.aad?.rememberedAccountsUrl)) ||
                        (accountType === AccountType.MSA && isStringOrFunction(this.config.msa?.rememberedAccountsUrl))
                    );
        }

        assertNever(authOperation, `Operation "${authOperation}" was unhandled.`);
    }

    public signIn(args: ISignInArgs): void {
        // Do nothing
    }

    public signInTo(args: ISignInToArgs): void {
        // Do nothing
    }

    public signOutFromApp(args: ISignOutFromAppArgs): void {
        // Do nothing
    }

    public signOutFromIdp(args: ISignOutFromIdpArgs): Promise<void> {
        const acctType = args.account.type;
        if (this.isOperationSupported('signOutFromIdp', acctType)) {
            switch (acctType) {
                case AccountType.AAD:
                case AccountType.MSA_FED:
                    return this.aadAccountsProvider.signOutFromIdp(args);
                case AccountType.MSA:
                    return this.msaAccountsProvider.signOutFromIdp(args);
            }
        }

        return Promise.reject(createError('signOutFromIdp is not supported'));
    }

    public signOutAndForgetFromIdp(args: ISignOutAndForgetFromIdpArgs): Promise<void> {
        const acctType = args.account.type;
        if (this.isOperationSupported('signOutAndForgetFromIdp', acctType)) {
            switch (acctType) {
                case AccountType.AAD:
                case AccountType.MSA_FED:
                    return this.aadAccountsProvider.signOutAndForgetFromIdp(args);
                case AccountType.MSA:
                    return this.msaAccountsProvider.signOutAndForgetFromIdp(args);
            }
        }

        return Promise.reject(createError('signOutAndForgetFromIdp is not supported'));
    }

    public switch(args: ISwitchArgs): void {
        // Do nothing
    }

    public switchTo(args: ISwitchToArgs): void {
        // Do nothing
    }

    public getRememberedAccounts(): Promise<IAccount[]> {
        const handleFailure = (type: AccountType) => (error: Error) => {
            logTelemetryEvent({
                eventType: 'ClientError',
                name: 'Failed to retrieve rememebered accounts for ' + type,
                type: 'RememberedAccountsFailure',
                details: JSON.stringify(toJsonable(error)),
                displayed: false
            });

            let empty: IAccount[] = [];
            return empty;
        };

        const msaSupported = this.isOperationSupported('getRememberedAccounts', AccountType.MSA);
        const aadSupported = this.isOperationSupported('getRememberedAccounts', AccountType.AAD);

        if (!msaSupported && !aadSupported) {
            return Promise.reject(createError('getRememberedAccounts is not supported'));
        }

        let msaRememberedAccounts = msaSupported ?
            this.msaAccountsProvider.getRememberedAccounts()
                .catch(handleFailure(AccountType.MSA))
                .then(accList => accList.filter(acc => acc.type == AccountType.MSA)) :
            Promise.resolve([]);

        let aadRememberedAccounts = aadSupported ?
            this.aadAccountsProvider.getRememberedAccounts()
                .catch(handleFailure(AccountType.AAD))
                .then(accList => accList.filter(acc => acc.type == AccountType.AAD)) :
            Promise.resolve([]);

        return Promise.all([msaRememberedAccounts, aadRememberedAccounts])
            .then(accountLists => accountLists.reduce((acc, cur) => acc.concat(cur), [])); // Flatten list
    }
}
