import { IWebMsaOnlyConfig, SignInExperienceType, IAccount, AccountType } from '@mecontrol/public-api';
import {
    IAuthProvider,
    IAuthOperations,
    IAuthNavigationProvider,
    ISignInArgs,
    ISignInToArgs,
    ISignOutFromAppArgs,
    ISignOutFromIdpArgs,
    ISignOutAndForgetFromIdpArgs,
    ISwitchArgs,
    ISwitchToArgs
} from '@mecontrol/common';
import { Promise, ME, assertNever, createError } from '@mecontrol/web-inline';
import { IRememberedAccountsProvider, createMsaAccountsProvider } from './webAccountProviders';
import { isStringOrFunction } from '../../../utilities';
import { canLeverageRememberedAccounts } from './operationHelpers';
import { WebAuthNavProvider } from './navProviders';

export function createWebMsaOnlyAuthProvider(config: IWebMsaOnlyConfig): IAuthProvider {
    return new WebMsaOnlyProvider(config);
}

class WebMsaOnlyProvider implements IAuthProvider {
    private msaAccountsProvider: IRememberedAccountsProvider;

    public navProvider: IAuthNavigationProvider;
    public supportedSignInAccountTypes: SignInExperienceType = SignInExperienceType.Msa;

    constructor(private config: IWebMsaOnlyConfig) {
        this.navProvider = new WebAuthNavProvider(this.config);
        this.msaAccountsProvider = createMsaAccountsProvider(this.config.msa);
    }

    public isOperationSupported(authOperation: keyof IAuthOperations, accountType: AccountType): boolean {
        if (accountType !== AccountType.MSA) {
            return false;
        }

        // TODO: AAD does this differently (seems to try to fix up for missing Switch URLs and SwitchToURLs.) Is that what we want?
        switch (authOperation) {
            case "signIn":
                return isStringOrFunction(this.config.appSignInUrl);
            case "signInTo":
                return isStringOrFunction(this.config.appSignInToUrl);
            case "signOutFromApp":
                return isStringOrFunction(this.config.appSignOutUrl);
            case "signOutFromIdp":
                return false; // MSA can't do this - it's implicitly sign out and forget
            case "signOutAndForgetFromIdp":
                return isStringOrFunction(this.config.msa?.signOutAndForgetUrl);
            case "switch":
                return isStringOrFunction(this.config.appSwitchUrl);
            case "switchTo":
                return isStringOrFunction(this.config.appSwitchToUrl);
            case "getRememberedAccounts":
                // TODO: aad also checks on ME.Config.aad, but there is no equivalent .msa property - why not?
                return ME.Config.remAcc &&
                    canLeverageRememberedAccounts(this, accountType) &&
                    isStringOrFunction(this.config.msa?.rememberedAccountsUrl); // ME.Config.msa - //TODO: there is a ME.Config.aad, but why not .msa?
        }

        assertNever(authOperation, `Operation "${authOperation}" was unhandled.`);
    }

    public signIn(args: ISignInArgs): void {
        // Do nothing
    }

    public signInTo(args: ISignInToArgs): void {
        // Do nothing
    }

    public signOutFromApp(args: ISignOutFromAppArgs): void {
        // Do nothing
    }

    public signOutFromIdp(args: ISignOutFromIdpArgs): Promise<void> {
        if (this.isOperationSupported('signOutFromIdp', args.account.type)) {
            return this.msaAccountsProvider.signOutFromIdp(args);
        }

        return Promise.reject(createError('signOutFromIdp is not supported'));
    }

    public signOutAndForgetFromIdp(args: ISignOutAndForgetFromIdpArgs): Promise<void> {
        if (this.isOperationSupported('signOutAndForgetFromIdp', args.account.type)) {
            return this.msaAccountsProvider.signOutAndForgetFromIdp(args);
        }

        return Promise.reject(createError('signOutAndForgetFromIdp is not supported'));
    }

    public switch(args: ISwitchArgs): void {
        // Do nothing
    }

    public switchTo(args: ISwitchToArgs): void {
        // Do nothing
    }

    public getRememberedAccounts(): Promise<IAccount[]> {
        if (this.isOperationSupported('getRememberedAccounts', AccountType.MSA)) {
            return this.msaAccountsProvider.getRememberedAccounts();
        }

        return Promise.reject(createError('getRememberedAccounts is not supported'));
    }
}
