import { Promise, ME, createError } from '@mecontrol/web-inline';
import { AuthProviderConfigType, IAuthProviderConfig, SignInExperienceType, AccountType } from '@mecontrol/public-api';
import { IAuthProvider, GetAuthProvider } from '@mecontrol/common';
import { createPartnerManagedAuthProvider } from './providers/partnerManaged';
import { createWebDualStackAuthProvider } from './providers/webDualStackProvider';
import { createWebMsaOnlyAuthProvider } from './providers/webMsaOnlyProvider';
import { createWebAadOnlyAuthProvider } from './providers/webAadOnlyProvider';
import { createWebAadWithMsaFedAuthProvider } from './providers/webAadWithMSAFedProvider';
import { createAdalJsAuthProvider } from './providers/adaljsProvider';
import { createV1BackCompatAuthProvider } from './providers/v1AuthProvider';
import { createMsalJsAuthProvider } from './providers/msaljsProvider';
import { createWebMsaWithAadProxyProvider } from './providers/webMsaWithAadProxy';
import { createWebAadWithMsaProxyProvider } from './providers/webAadWithMsaProxy';

export { isRememberedAccountsSupported } from './providers/operationHelpers';

type AuthProviderFactory<T extends IAuthProviderConfig> = (config: T) => IAuthProvider;

type AuthProviderConfigToFactoryMap = {
    [key in AuthProviderConfigType]: AuthProviderFactory<any> // TODO: Try making this
};

const authProviderConfigToFactoryMap: AuthProviderConfigToFactoryMap = {
    [AuthProviderConfigType.PartnerManaged]: createPartnerManagedAuthProvider,
    [AuthProviderConfigType.WebDualStack]: createWebDualStackAuthProvider,
    [AuthProviderConfigType.WebMsaOnly]: createWebMsaOnlyAuthProvider,
    [AuthProviderConfigType.WebAadOnly]: createWebAadOnlyAuthProvider,
    [AuthProviderConfigType.WebAadWithMsaFed]: createWebAadWithMsaFedAuthProvider,
    [AuthProviderConfigType.WebAadWithMsaProxy]: createWebAadWithMsaProxyProvider,
    [AuthProviderConfigType.WebMsaWithAadProxy]: createWebMsaWithAadProxyProvider,
    [AuthProviderConfigType.AdalJsWithMsaFed]: createAdalJsAuthProvider,
    [AuthProviderConfigType.AdalJsAadOnly]: createAdalJsAuthProvider,
    [AuthProviderConfigType.MsalJsAadOnly]: createMsalJsAuthProvider,
    [AuthProviderConfigType.MsalJsWithMsa]: createMsalJsAuthProvider,
    [AuthProviderConfigType.V1BackCompat]: createV1BackCompatAuthProvider
};

export const getAuthProvider: GetAuthProvider = options => {
    const configType = options.authProviderConfig.type;

    let authProvider: IAuthProvider;
    if (configType in authProviderConfigToFactoryMap) {
        authProvider = authProviderConfigToFactoryMap[configType](options.authProviderConfig);
    }
    else {
        throw createError(`No AuthProvider factory matches given AuthProvider config type: ${configType}`);
    }

    // TODO: Consider a separate flag for web and native
    // TODO: Put this check in each auth providers isOperationSupported check
    if (ME.Config.remAcc == false) {
        authProvider.getRememberedAccounts = () => Promise.reject(createError('RememberedAccountsDisabled'));
    }

    return authProvider;
};

/**
 * Converts the given supported account type flags to a usable list object.
 * @param accountTypes The flag enum value to conver to a list.
 */
export function supportedAccountTypesToList(accountTypes: SignInExperienceType): SignInExperienceType[] {

    let supportedTypes: SignInExperienceType[] = [];
    if (accountTypes !== SignInExperienceType.Converged) {
        // Not default, result can be either option or both.
        if ((accountTypes & SignInExperienceType.Aad) == SignInExperienceType.Aad) {
            supportedTypes.push(SignInExperienceType.Aad);
        }

        if ((accountTypes & SignInExperienceType.Msa) == SignInExperienceType.Msa) {
            supportedTypes.push(SignInExperienceType.Msa);
        }
    }
    else {
        // Default
        supportedTypes.push(SignInExperienceType.Converged);
    }

    return supportedTypes;
}

/**
 * Converts an AccountType enum into a valid SignInExperienceType enum value.
 * @param accountType The AccountType to be converted.
 */
export function getSupportedAccountType(accountType: AccountType): SignInExperienceType {
    switch (accountType) {
        case AccountType.AAD:
            return SignInExperienceType.Aad;

        default:
            return SignInExperienceType.Msa;
    }
}
