import {
    AccountType,
    IAccount,
    AuthenticatedState,
} from '@mecontrol/public-api';
import { ErrorSeverity } from '@mecontrol/common';
import {
    IUserData,
    IAadUserData,
    IMsaUserData,
    UserDataAuthState,
    UserDataType
} from './userData';
import { logTelemetryEvent } from '@mecontrol/web-inline';
import { decodeHtml } from '../../utilities';

type UserDataTypeMap = { [key in UserDataType]: AccountType };
const UserDataTypeMap: UserDataTypeMap = {
    [UserDataType.AAD]: AccountType.AAD,
    [UserDataType.MSA]: AccountType.MSA,
    [UserDataType.MSA_FED]: AccountType.MSA_FED,
};

interface UserDataAuthStateMap {
    [key: number]: AuthenticatedState;
}

const UserDataAuthStateMap: UserDataAuthStateMap = {
    [UserDataAuthState.NotSignedIn]: AuthenticatedState.NotSignedIn,
    [UserDataAuthState.SignedIn]: AuthenticatedState.SignedIn,
    [UserDataAuthState.SignedInIdp]: AuthenticatedState.SignedInIDPOnly,
};

export function userDataToIAccount(userData: IUserData | IAadUserData | IMsaUserData): IAccount {
    const msaUserData = userData as IMsaUserData;
    const aadUserData = userData as IAadUserData;

    // If an IDP tells us an account is signed in to the site, we map the state
    // back to IDP-only to avoid issues with mismatched states after a sign out
    // from app or switch.
    // This only applies to website remembered accounts and will probably need
    // to be addressed again in the future when a website actually supports more
    // than one account in a 'signed in to app' state.
    const filterAuthenticatedStatus = (authStatus: AuthenticatedState) =>
        authStatus === AuthenticatedState.SignedIn ?
            AuthenticatedState.SignedInIDPOnly :
            authStatus;

    let authState = UserDataAuthStateMap[userData.authenticatedState];
    if (!authState) {
        authState = UserDataAuthStateMap[UserDataAuthState.NotSignedIn];
        logTelemetryEvent({
            eventType: "ClientError",
            name: "Undefined Authenticated State",
            type: "UndefinedAuthState",
            details: `An unsupported authenticated state was supplied: ${userData.authenticatedState}`,
            severity: ErrorSeverity.Warning,
            displayed: false
        });
    }

    return {
        type: UserDataTypeMap[userData.idp] as any,
        memberName: decodeHtml(userData.memberName) as string,
        authenticatedState: filterAuthenticatedStatus(authState),
        firstName: decodeHtml(userData.firstName),
        lastName: decodeHtml(userData.lastName),
        pictureUrl: userData.tileUrl,
        cid: msaUserData.cid,
        hA: msaUserData.hA,
        displayName: decodeHtml(aadUserData.displayName),
        orgName: decodeHtml(aadUserData.orgName),
        orgLogoUrl: aadUserData.orgLogoUrl,
        sessionId: aadUserData.sessionId,
        roleName: decodeHtml(aadUserData.roleName),
        isWindowsSso: aadUserData.isWindowsSso,
    };
}
