import { h, FunctionalComponent } from 'preact';
import cc from 'classcat';
import { AuthenticatedState } from '@mecontrol/public-api';
import {
    IAccountState,
    IAuthProvider,
    IAuthActions,
    PageAction,
    AccountItemStatus,
} from '@mecontrol/common';
import { getString, createId } from '@mecontrol/web-inline';
import {
    ProfilePicture,
    LinkButton,
    getAccountListDisplayName,
    isSignedIn,
    format,
    getVerbalFriendlyName,
    isAadLike
} from '@mecontrol/web-boot';
import { AccountInfo } from './AccountInfo';
import { AccountActions } from './AccountActions';
import { IMenuItemProps } from '../Menu';

export interface IAccountItemProps {
    index: number;
    account: IAccountState;
    authProvider: IAuthProvider;
    hasActiveAccount: boolean;
    onItemClick: (account: IAccountState, event: Event) => void;
    getItemUrl: (account: IAccountState) => string | undefined;
    signOutFromApp: IAuthActions["signOutFromApp"];
    signOutFromIdp: IAuthActions["signOutFromIdp"];
    signOutAndForgetFromIdp: IAuthActions["signOutAndForgetFromIdp"];
}

export interface IAccountItemState { }

// Types used to generate account actions
type SignOutOperation = 'signOutFromApp' | 'signOutFromIdp' | 'signOutAndForgetFromIdp';
type SignOutUrlGetter = 'getSignOutFromAppUrl' | 'getSignOutFromIdpUrl' | 'getSignOutAndForgetFromIdpUrl';
type OperationValue = Record<AuthenticatedState, Partial<Record<SignOutOperation, string>>>;
type OperationUrlGetter = Record<SignOutOperation, SignOutUrlGetter>;

const stringKeys: OperationValue = {
    signedIn: { signOutFromApp: 'signout' },
    signedInIDPOnly: { signOutFromIdp: 'signout', signOutAndForgetFromIdp: 'signoutandforget' },
    notSignedIn: { signOutAndForgetFromIdp: 'forget' }
};

const ariaStringKeys: OperationValue = {
    signedIn: { signOutFromApp: 'signoutaccounta11yformat1' },
    signedInIDPOnly: { signOutFromIdp: 'signoutaccounta11yformat1', signOutAndForgetFromIdp: 'signoutandforgetaccounta11yformat1' },
    notSignedIn: { signOutAndForgetFromIdp: 'forgetaccounta11yformat1' }
};

const pageAction: OperationValue = {
    signedIn: { signOutFromApp: PageAction.signOut },
    signedInIDPOnly: { signOutFromIdp: PageAction.accountSignOut, signOutAndForgetFromIdp: PageAction.accountSignOutAndForget },
    notSignedIn: { signOutAndForgetFromIdp: PageAction.accountForget }
};

const urlGetterName: OperationUrlGetter = {
    signOutFromApp: 'getSignOutFromAppUrl',
    signOutFromIdp: 'getSignOutFromIdpUrl',
    signOutAndForgetFromIdp: 'getSignOutAndForgetFromIdpUrl',
};

// Used to create IDs
const mainIdPart = 'rememberedAccount';

export const AccountItem: FunctionalComponent<IAccountItemProps> = props => {
    const { index, account, authProvider, hasActiveAccount, onItemClick, getItemUrl } = props;

    const itemState = account.accountItemStatus;
    const isLoading = itemState === AccountItemStatus.Loading;

    const friendlyNameA11y = getVerbalFriendlyName(account);

    /**
     * Handler that tries to sign out the item's account
     * @param action Whether the handler is signing out or signing out and forgetting
     * @param event Mouse event
     */
    function onSignOutActionClick(operation: SignOutOperation, event: MouseEvent): void {
        // Only try signing out if we are not already trying it
        if (account.accountItemStatus !== AccountItemStatus.Loading) {
            // Use Auth API to sign out the account
            if (operation === 'signOutFromApp') {
                props[operation]({ currentAccount: account }, event);
            } else {
                props[operation]({ account }, event);
            }
        }
        else {
            event.preventDefault();
        }
    }

    /**
     * Generate the properties for an account action
     * @param operation Which account action to generate properties for
     */
    function getAccountAction(operation: SignOutOperation): IMenuItemProps | undefined {
        const navProvider = authProvider.navProvider;

        if (authProvider.isOperationSupported(operation, account.type)) {
            const getUrl = () => {
                let methodName = urlGetterName[operation];
                if (methodName === 'getSignOutFromAppUrl') {
                    return navProvider && navProvider[methodName] && navProvider[methodName]({ currentAccount: account });
                } else {
                    return navProvider && navProvider[methodName] && navProvider[methodName]({ account });
                }
            };

            return {
                id: createId(mainIdPart, index, operation),
                contentId: pageAction[account.authenticatedState][operation],
                contentSlot: index,
                key: operation,
                isDisabled: isLoading,
                getUrl,
                onClick: onSignOutActionClick.bind(null, operation),
                children: [getString(stringKeys[account.authenticatedState][operation]!)],
                ariaLabel: format(getString(ariaStringKeys[account.authenticatedState][operation]!), getVerbalFriendlyName(account))
            };
        }
    }

    /**
     * Generate an array of IAccountActionProps for actions supported by this account item
     * @param allowSignOut Whether the Sign Out action is allowed (i.e. the account is signed in)
     */
    function getAccountActionProps(): IMenuItemProps[] {
        let accountActionProps: (IMenuItemProps | undefined)[];
        if (isAadLike(account) && account.isWindowsSso) {
            return [];
        }
        else if (account.authenticatedState === AuthenticatedState.SignedIn) {
            accountActionProps = [
                getAccountAction('signOutFromApp')
            ];
        } else if (account.authenticatedState === AuthenticatedState.SignedInIDPOnly) {
            accountActionProps = [
                getAccountAction('signOutFromIdp'),
                getAccountAction('signOutAndForgetFromIdp')
            ];
        } else {
            accountActionProps = [
                getAccountAction('signOutAndForgetFromIdp')
            ];
        }

        return accountActionProps.filter(Boolean) as IMenuItemProps[];
    }

    const accountActionItems = isLoading ? [] : getAccountActionProps();

    return (
        <div class='mectrl_accountItem'>
            <LinkButton
                id={createId(mainIdPart, index, 'switch')}
                contentId={PageAction.accountSwitch}
                contentType={account.type}
                contentSlot={index}
                isDisabled={isLoading}
                getUrl={() => getItemUrl(account)}
                onClick={ev => onItemClick(account, ev)}
                cssClass={cc(['primaryAction', hasActiveAccount ? 'switchTo' : 'signInTo'])}
                ariaLabel={getAccountSwitchAriaLabel(account)}
            >
                <ProfilePicture
                    id={createId(mainIdPart, index, 'picture')}
                    currentAccount={account}
                />
                <AccountInfo
                    id={createId(mainIdPart, index)}
                    friendlyName={getAccountListDisplayName(account)}
                    details={getAccountInfoDetails(account)}
                    isHighlighted={isSignedIn(account)}
                />
            </LinkButton>
            {accountActionItems.length > 0 ?
                <AccountActions
                    id={createId(mainIdPart, index, 'actions')}
                    index={index}
                    items={accountActionItems}
                    triggerAriaLabel={format(getString('signoutmenua11yformat1'), friendlyNameA11y)}
                /> :
                null
            }
        </div>
    );
};

function getAccountInfoDetails(account: IAccountState): string {
    switch (account.accountItemStatus) {
        case AccountItemStatus.Normal:
            return account.memberName;
        case AccountItemStatus.Loading:
            return getString('signoutloading');
        case AccountItemStatus.Error:
            return getString('signouterror');
    }
}

function getAccountSwitchAriaLabel(account: IAccountState): string {
    const friendlyName = getVerbalFriendlyName(account);
    let switchMsg = format(getString('switchaccounta11yformat1'), friendlyName);

    switch (account.accountItemStatus) {
        case AccountItemStatus.Normal:
            return switchMsg;

        case AccountItemStatus.Loading:
        case AccountItemStatus.Error:
            const lastChar = switchMsg[switchMsg.length - 1];
            switchMsg += lastChar === '.' ? '' : '.';
            let statusMsg = format(getString('accountstatusa11yformat1'), getAccountInfoDetails(account));

            return `${switchMsg} ${statusMsg}`;
    }
}
