import { h, FunctionalComponent } from 'preact';
import cc from 'classcat';
import { AuthenticatedState, SignInExperienceType, AccountType } from '@mecontrol/public-api';
import { IMeControlAccounts, IAccountState, IAuthProvider, IAuthActions, ISwitchArgs, IAuthOperations } from '@mecontrol/common';
import { getOtherAccounts, getAccountId, getCurrentAccount } from '@mecontrol/web-boot';
import { AccountItem } from './AccountItem';
import { SignInItem } from './SignInItem';
import { ScrollShadow } from './ScrollShadow';
import { getString } from '@mecontrol/web-inline';

export interface IAccountListProps {
    accounts: IMeControlAccounts;
    authProvider: IAuthProvider;
    switchTo: IAuthActions["switchTo"];
    switch: IAuthActions["switch"];
    signInTo: IAuthActions["signInTo"];
    signIn: IAuthActions["signIn"];
    signOutFromApp: IAuthActions["signOutFromApp"];
    signOutFromIdp: IAuthActions["signOutFromIdp"];
    signOutAndForgetFromIdp: IAuthActions["signOutAndForgetFromIdp"];
}

// TODO: Pato -  move all account list related components into sub-folder to match CSS folder stucture.
export const AccountList: FunctionalComponent<IAccountListProps> = props => {
    const {
        accounts,
        authProvider,
        switchTo,
        signInTo,
        signIn,
        signOutFromApp,
        signOutFromIdp,
        signOutAndForgetFromIdp,
    } = props;

    const accountsToDisplay = getOtherAccounts(accounts);
    const currentAccount = getCurrentAccount(accounts);
    const hasActiveAccount = currentAccount !== undefined;

    /**
     * Callback method used to fire the switch or signIn callback that may be defined by
     * the onboarded client/partner page or app.
     * @param signInType The SignInExperienceType that is being switched or signed in.
     * @param event The event that fired the action (example: MouseClickEvent).
     */
    const switchOrSignInClick = (signInType: SignInExperienceType, event: Event) => {
        if (hasActiveAccount) {
            props.switch({ currentAccount, signInType } as ISwitchArgs, event);
        }
        else {
            signIn({ signInType }, event);
        }
    };

    /**
     * Callback method used to get the switch or signIn url from the auth provider.
     * @param signInType The SignInExperienceType that is being requested.
     */
    const getSwitchOrSignInUrl = (signInType: SignInExperienceType) => {
        if (authProvider.navProvider) {
            if (hasActiveAccount) {
                return authProvider.navProvider.getSwitchUrl({ currentAccount, signInType } as ISwitchArgs);
            }
            else {
                return authProvider.navProvider.getSignInUrl({ signInType });
            }
        }
    };

    /**
     * Callback method used to fire the switchTo or SignInTo callback that may be defined by
     * the onboarded client/partner page or app.
     * @param account The IAccountState that is being switched to or signed in to.
     * @param event The event that fired the action (example: MouseClickEvent).
     */
    const switchToOrSignInTo = (account: IAccountState, event: Event) => {
        if (currentAccount) {
            switchTo({
                currentAccount,
                nextAccount: account
            }, event);
        }
        else {
            signInTo({ nextAccount: account }, event);
        }
    };

    /**
     * Callback method used to get the signInTo or switchTo url from the auth provider.
     * @param account The IAccountState that is going to be signed in to or switched to.
     */
    const getSignInToOrSwitchToUrl = (account: IAccountState) => {
        if (authProvider.navProvider) {
            if (currentAccount) {
                return authProvider.navProvider.getSwitchToUrl({
                    currentAccount,
                    nextAccount: account,
                });
            }
            else {
                return authProvider.navProvider.getSignInToUrl({ nextAccount: account });
            }
        }
    };

    /**
     * Callback method used to filter items that will be displayed in the remembered accounts list.
     * This method basically evaluates the isOperationSupported before displaying the item.
     * @param account The remembered account to be displayed.
     */
    const shouldDisplay = (account: IAccountState) => {
        return (hasActiveAccount && authProvider.isOperationSupported('switchTo', account.type))
            || (!hasActiveAccount && authProvider.isOperationSupported('signInTo', account.type));
    };

    // Gets the initial remembered account list to display (if supported).
    let accountItems = accountsToDisplay
        .filter(shouldDisplay)
        .sort(accountComparer)
        .map((account, index) =>
            <li key={getAccountId(account)}>
                <AccountItem
                    index={index}
                    account={account}
                    hasActiveAccount={hasActiveAccount}
                    authProvider={authProvider}
                    onItemClick={switchToOrSignInTo}
                    getItemUrl={getSignInToOrSwitchToUrl}
                    signOutFromApp={signOutFromApp}
                    signOutFromIdp={signOutFromIdp}
                    signOutAndForgetFromIdp={signOutAndForgetFromIdp}
                />
            </li>
        );

    // TODO: trevori/andwi: This is supported op will need to change some day when we can pass in SignInExperienceType.
    // Append the "sign-in" item to the account list
    if (!hasActiveAccount || isSupportedForAny(authProvider, 'switch')) {
        accountItems.push(
            <li>
                <SignInItem
                    authProvider={authProvider}
                    signInClick={switchOrSignInClick}
                    getSignInUrl={getSwitchOrSignInUrl}
                />
            </li>
        );
    }

    return (
        <ScrollShadow cssClass={cc(['mectrl_accountList_container', { signedOut: !hasActiveAccount } ])}>
            <ul aria-label={!hasActiveAccount ? getString('chooseanaccount') : undefined} class='mectrl_accountList'>
                {accountItems}
            </ul>
        </ScrollShadow>
    );
};

/**
 * Calls the auth provider isOperationSupport for the given operation type against all AccountType options to
 * ensure the operations can works for any of the account types. A single success returns TRUE.
 * @param authProvider The current auth provider.
 * @param operation The auth operation type to check.
 */
function isSupportedForAny(authProvider: IAuthProvider, operation: keyof IAuthOperations): boolean {
    return authProvider.isOperationSupported(operation, AccountType.MSA)
        || authProvider.isOperationSupported(operation, AccountType.AAD)
        || authProvider.isOperationSupported(operation, AccountType.MSA_FED);
}

/**
 * Comparer function for sorting accounts. Sort order is based on authenticated
 * state as follows:
 * - SignedIn
 * - SignedInIDPOnly
 * - NotSignedIn
 * Within the same auth state, sort order is determined by member name alphabetically
 * @param A First account to compare
 * @param B Second account to compare
 * @returns A number with the folowwing meaning:
 * - -1 means A comes first
 * - 0 means they are the same
 * - 1 means B comes first
 */
function accountComparer(A: IAccountState, B: IAccountState): number {
    if (A.authenticatedState === B.authenticatedState) {
        let aMN = A.memberName.toLowerCase();
        let bMN = B.memberName.toLowerCase();
        if (aMN < bMN) {
            return -1;
        }
        else if (aMN > bMN) {
            return 1;
        }
        else {
            return 0;
        }
    }
    else {
        if (A.authenticatedState === AuthenticatedState.SignedIn) {
            return -1;
        }
        else if (A.authenticatedState === AuthenticatedState.NotSignedIn) {
            return 1;
        }
        else {
            if (B.authenticatedState === AuthenticatedState.SignedIn) {
                return 1;
            }
            else {
                return -1;
            }
        }
    }
}
