import { h, FunctionalComponent } from 'preact';
import cc from 'classcat';
import { SignInExperienceType } from '@mecontrol/public-api';
import { AnchorPosition, AnchorAlignment, PageAction, IAuthProvider } from '@mecontrol/common';
import { getString, createId } from '@mecontrol/web-inline';
import { ProfilePicture, LinkButton, TruncateText } from '@mecontrol/web-boot';
import { IMenuItemProps, Menu, fixedMenuClass } from '../Menu';

/**
 * The props required for the SignInItem to render appropriately.
 */
export interface ISignInItemProps {
    /**
     * The IAuthProvider set by the partner.
     */
    authProvider: IAuthProvider;

    /**
     * Callback method that is triggered when the user clicks this sign-in control. This must be provided
     * in order for the parent componenet to determine the current signed in state of the MeControl in
     * order to know if sign in vs. switch should be used.
     */
    signInClick: (signInType: SignInExperienceType, event: Event) => void;

    /**
     * Callback method that is executed upon rendering of this component to get the sign in or switch
     * url from the auth provider. This must be provided in order for the parent componenet to determine
     * the current signed in state of the MeControl in order to know if sign in vs. switch should be used.
     */
    getSignInUrl: (signInType: SignInExperienceType) => string | undefined;
}

const mainIdPart = 'signInItem';
const cssClassForLinkButton = 'primaryAction signIn';

/**
 * Component that appears following the remembered account list which allows the user to sign in (or switch) to
 * another account. This componenet will determine the appropriate supported provider types and either display
 * an inline button for a seamless experience or a drop down allowing the user to select the desired account
 * type to sign in or switch to.
 * @param props The appropriate ISignInItemProps used to render the proper component.
 */
export const SignInItem: FunctionalComponent<ISignInItemProps> = props => {
    const { authProvider } = props;

    if (authProvider.supportedSignInAccountTypes === (SignInExperienceType.Msa | SignInExperienceType.Aad)) {
        return <MenuSignInItem {...props} />;
    }
    else {
        return <SimpleSignInItem {...props} />;
    }
};

interface SignInMenuItemGeneratorProps {
    index: number;
    key: string;
    text: string;
    signInType: SignInExperienceType;
}
type SignInMenuItemGenerator = (props: SignInMenuItemGeneratorProps) => IMenuItemProps;

/**
 * Menu whose trigger mimics the HTML structure and classes of the AccountItem component
 * in order to share styling
 */
const MenuSignInItem: FunctionalComponent<ISignInItemProps> = props => {
    const { signInClick, getSignInUrl } = props;

    const getItemProps: SignInMenuItemGenerator = itemProps => ({
        id: createId(mainIdPart, itemProps.index),
        key: itemProps.key,
        contentId: PageAction.accountSignIn,
        contentSlot: itemProps.index,
        getUrl: () => getSignInUrl(itemProps.signInType),
        onClick: event => signInClick(itemProps.signInType, event),
        children: [itemProps.text]
    });

    return (
        <Menu
            id={createId(mainIdPart)}
            contentId={PageAction.accountSignInMenu}
            trigger={{
                children: getSignInItemContents(),
                cssClass: cssClassForLinkButton,
                ariaLabel: getString('signinaddaccounta11y')
            }}
            items={[
                getItemProps({
                    index: 0,
                    key: 'signInWithMSAItem',
                    text: getString('signinwithmsaaccount'),
                    signInType: SignInExperienceType.Msa
                }),
                getItemProps({
                    index: 1,
                    key: 'signInWithAADItem',
                    text: getString('signinwithaadaccount'),
                    signInType: SignInExperienceType.Aad
                })
            ]}
            cssClass={cc(['mectrl_accountItem', fixedMenuClass])}
            position={AnchorPosition.Top}
            alignment={AnchorAlignment.End}
        />
    );
};

/**
 * Sign In Item that is a simple LinkButton emulating the same HTML structure and classes
 * as an AccountItem (in order to share styling)
 */
const SimpleSignInItem: FunctionalComponent<ISignInItemProps> = props => {
    const { authProvider, signInClick, getSignInUrl } = props;

    return (
        <div class='mectrl_accountItem'>
            <LinkButton
                id={createId(mainIdPart)}
                contentId={PageAction.accountSignIn}
                getUrl={() => getSignInUrl(authProvider.supportedSignInAccountTypes)}
                onClick={event => signInClick(authProvider.supportedSignInAccountTypes, event)}
                cssClass={cssClassForLinkButton}
                ariaLabel={getString('signinwithdifferentaccount')}
            >
                {getSignInItemContents()}
            </LinkButton>
        </div>
    );
};

/** Generate shared contents for the SignInItem LinkButton */
function getSignInItemContents(): JSX.Element[] {
    return [
        <ProfilePicture
            id={createId(mainIdPart, 'picture')}
            glyphClassOverride='mectrl_signIn_circle_glyph'
        />,
        <div class='mectrl_accountItemInfo'>
            <TruncateText text={getString('signinwithdifferentaccount')} />
        </div>
    ];
}
