import { render, h } from 'preact';
// import "preact/debug"; // Preact debug is wrapped in a NODE_ENV check and so is only included in debug builds
import { IPublicApiProxy } from '@mecontrol/public-api';
import { PageName, Scenario, ScenarioStage, ContentSource } from '@mecontrol/common';
import { MainInit, ME, loadCss, createError, logTelemetryEvent, getTimeSinceStart } from '@mecontrol/web-inline';
import { Root } from './components/Root';
import { configureStore } from './store/configureStore';
import { Provider } from './core/Provider';
import { initOptions } from './actions/initOptions';
import { getRememberedAccounts } from './actions/authActions';
import { createController } from './publicApiController';
import { getAuthProvider } from './api/auth';
import { onLoadUntilUrgent, delayUntilAnimationFrame, loadCore } from './utilities';
import { hydrateFromCache } from './actions/cacheActions';

export { h, Component, FunctionalComponent } from 'preact';
export * from './core/Provider';
export * from './core/connect';
export * from './core/toChildArray';
export * from './utilities';
export * from './components/ProfilePicture';
export * from './components/ProfilePictureOverlay';
export * from './components/LinkButton';
export * from './components/Tooltip';
export * from './components/TruncateText';
export * from './components/ScreenReaderText';
export * from './components/FocusVisible';
export * from './components/AuthAppUpsell';
export * from './actions/presenceActions';

export * from './api/auth';

export const init: MainInit = (options, setController) => {
    if (!options) {
        throw createError("Options required");
    }

    const container = options.shadowDOMContainerElement || document.getElementById(options.containerId);
    if (!container) {
        throw createError(`Could not find element with id ${options.containerId}`);
    }

    while (container.firstChild) {
        container.removeChild(container.firstChild);
    }

    loadCss(ME.CssmeBoot);

    const authProvider = getAuthProvider(options);
    const syntheticEventTarget = options.syntheticEventTarget;

    const store = configureStore({ authProvider, syntheticEventTarget });

    store.dispatch(initOptions(options));

    // setup the new controller, but keep a handle to it because...
    let publicApiProxy: IPublicApiProxy = createController(store, syntheticEventTarget);
    setController(publicApiProxy);

    // TODO: add a prop for the SyntheticEventTarget to pass down so that we can use it later in Root for expand/collapse events
    render(
        <Provider store={store}>
            <Root authProvider={authProvider} />
        </Provider>,
        container
    );

    logTelemetryEvent({
        eventType: 'ContentUpdate',
        content: {
            id: PageName.initial,
            source: ContentSource.UX
        },
        loadTime: getTimeSinceStart()
    });

    onLoadUntilUrgent().then(() => {
        store.dispatch(getRememberedAccounts()).then(() => {
            store.dispatch(hydrateFromCache());
        });
    });

    loadCore().then(meCore => {
        meCore.enhanceTelemetry();
    });

    delayUntilAnimationFrame(() => {
        logTelemetryEvent({
            eventType: 'TrackedScenario',
            scenario: Scenario.Interactive,
            previousAction: ScenarioStage.Start,
            action: ScenarioStage.End,
            success: true,
            durationMs: getTimeSinceStart(),
            details: 'Web header'
        });
    });
};
