import { IAccount, ICommand, IPublicApiProxy, MeControlEventType, RuntimeDisplayMode, ITheme, IPresenceConfig, IPresenceState } from '@meControl/public-api';
import { MeControlStore, ISyntheticEventTarget } from '@mecontrol/common';
import {
    setActiveAccount, setDisplayMode, setTheme, setCommands,
    setProfilePicture, setAccountPicture, removeActiveAccount, setHeaderTheme, refreshPicture
} from './actions/publicApiActions';
import { getCurrentAccount, getAccountId } from '../src/utilities/accounts';
import { getRememberedAccounts } from './actions/authActions';
import { setPresence, setPresenceConfig } from './actions/presenceActions';
import { setQueryParams, getCookie } from './utilities';

export function createController(store: MeControlStore, eventTarget: ISyntheticEventTarget): IPublicApiProxy {
    const currentAccount = () => getCurrentAccount(store.getState().accounts);

    return {
        setActiveAccount(activeAccount: IAccount | null): void {
            if (activeAccount) {
                store.dispatch(setActiveAccount(activeAccount));
            }
            else {
                const account = currentAccount();
                if (account) {
                    store.dispatch(removeActiveAccount(getAccountId(account)));
                }
            }

            // Fire off getting remembered accounts to get the latest state
            // and stay up to date
            store.dispatch(getRememberedAccounts());
        },

        setCommands(commands: ICommand[]): void {
            store.dispatch(setCommands(commands));
        },

        setAccountPicture(newUrl: string): void {
            const account = currentAccount();
            if (account) {
                store.dispatch(setAccountPicture(newUrl, getAccountId(account)));
            }
        },

        setProfilePicture(newUrl: string): void {
            const account = currentAccount();
            if (account) {
                store.dispatch(setProfilePicture(newUrl, getAccountId(account)));
            }
        },

        refreshAccountPicture(breakCache?: boolean): void {
            const account = currentAccount();

            if (account && account.pictureUrl) {
                if (breakCache) {
                    const newPictureUrl = setQueryParams(account.pictureUrl, {ck: getCookie('upck') || `${Math.floor(Math.random() * 100000000)}`});
                    store.dispatch(setAccountPicture(newPictureUrl, getAccountId(account)));
                } else {
                    store.dispatch(refreshPicture(account.pictureUrl, 'refreshAccountPicture()'));
                }
            }
        },

        refreshProfilePicture(breakCache?: boolean): void {
            const account = currentAccount();

            if (account && account.profile && account.profile.profilePictureUrl) {
                if (breakCache) {
                    const newPictureUrl = setQueryParams(account.profile.profilePictureUrl, {ck: getCookie('upck') || `${Math.floor(Math.random() * 100000000)}`});
                    store.dispatch(setProfilePicture(newPictureUrl, getAccountId(account)));
                } else {
                    store.dispatch(refreshPicture(account.profile.profilePictureUrl, 'refreshProfilePicture()'));
                }
            }
        },

        setTheme(theme: ITheme): void {
            store.dispatch(setTheme(theme));
            store.dispatch(setHeaderTheme(theme));
        },

        setHeaderTheme(headerTheme: ITheme): void {
            store.dispatch(setHeaderTheme(headerTheme));
        },

        setDisplayMode(mode: RuntimeDisplayMode): void {
            store.dispatch(setDisplayMode(mode));
        },

        loadPresenceConfig(config: IPresenceConfig): void {
            store.dispatch(setPresenceConfig(config));
        },

        setPresence(presence: IPresenceState): void {
            store.dispatch(setPresence(presence));
        },

        addEventListener(type: MeControlEventType, callback: any): void {
            eventTarget.addEventListener(type, callback);
        },

        removeEventListener(type: MeControlEventType, callback: any): void {
            eventTarget.removeEventListener(type, callback);
        }
    };
}
