import { Reducer } from 'redux';
import { IAccount } from '@meControl/public-api';
import { AllActions, ActionTypes } from '@mecontrol/common';
import { getAccountId } from '../../utilities/accounts';

/**
 * Reducer function to handle the array from the accounts state containing all the account IDs
 * @param state Array of account IDs
 * @param action Dispatched action to change the state
 * @returns The new state given the action
 */
export const allIds: Reducer<string[]> = (state = [], action: AllActions) => {
    switch (action.type) {

        // When initializing options, we create a new array of IDs, and only
        // add one for the active account if it is provided
        case ActionTypes.INIT_OPTIONS:
            const currentAccount = action.payload.options.currentAccount;
            if (currentAccount) {
                return addAccountIds(state, [currentAccount]);
            }

            return state;

        // When setting the active account, we only add the account's ID if it is
        // not already present in the array
        case ActionTypes.SET_ACTIVE_ACCOUNT:
            return addAccountIds(state, [action.payload.account]);

        // Remove the currently active account
        // Effect is the same as a succesful signout and forget
        case ActionTypes.REMOVE_ACTIVE_ACCOUNT:
            return removeAccountId(state, action.payload.accountId);


        // Add the remembered accounts' Ids that were recevied from the API
        case ActionTypes.REMEMBERED_ACCOUNTS_SUCCESS:
            return addAccountIds(state, action.payload.accounts);

        // Remove a signed out account's Id (if we need to forget it)
        case ActionTypes.SIGNOUT_ACCOUNT_SUCCESS:
            if (action.payload.forget) {
                return removeAccountId(state, action.payload.accountId);
            }
            else {
                return state;
            }

        default:
            return state;
    }
};

/**
 * Add an array of accounts's Ids to an array of account Ids
 * @param state Array of current account Ids
 * @param accounts Array of accounts whose Ids will be added to the current array
 * @returns A new array of account Ids with those from the accounts added (and de-duped)
 */
function addAccountIds(state: string[], accounts: IAccount[]): string[] {
    let newAccountIds = accounts
        // Convert accounts to Ids
        .map(account => getAccountId(account))
        // Dedupe Ids
        .reduce((dedupedIds: string[], accountId: string) => {
            if (dedupedIds.indexOf(accountId) === -1 && state.indexOf(accountId) === -1) {
                dedupedIds.push(accountId);
            }
            return dedupedIds;
        }, []);

    if (newAccountIds.length > 0) {
        return [
            ...state,
            ...newAccountIds
        ];
    }
    else {
        return state;
    }
}

/**
 * Remove the provided account's Id from the array of account Ids
 * @param accountIds Array of current account Ids
 * @param accountId Id of the Account to remove
 * @returns A new array of account Ids without the passed-in account's Id
 */
export function removeAccountId(accountIds: string[], accountId: string): string[] {
    return accountIds.filter(accId => accId !== accountId);
}
