import { Reducer } from 'redux';
import { IAccount, AuthenticatedState, AccountType, IAccountPropTypes } from '@meControl/public-api';
import {
    IAccountStateMap, IAccountState, AccountItemStatus, AllActions, ActionTypes, DeepPartial, IGraphData
} from '@mecontrol/common';
import { getAccountId, merge, createNewStateGuid } from '../../utilities';

/**
 * Reducer function to handle the object from the accounts state containing all Account information (account map)
 * @param state Map of accounts keyed by their computed IDs
 * @param action Dispatched action to change the state
 * @returns The new state given the action
 */
export const byId: Reducer<IAccountStateMap> = (state = {}, action: AllActions) => {

    switch (action.type) {

        // When initializing with options, we create a new map of accounts and,
        // if available, add the active account to it
        case ActionTypes.INIT_OPTIONS:
            const currentAccount = action.payload.options.currentAccount;

            if (currentAccount) {
                if (currentAccount?.uaid == null) {
                    currentAccount.uaid = createNewStateGuid();
                }

                return addAccounts(state, [currentAccount]);
            }

            return state;

        // When setting the active account, we only add the account to the map if
        // it was not already there
        // Additionally, we reset the account's accountItemState
        case ActionTypes.SET_ACTIVE_ACCOUNT:
            if (action.payload.account.uaid == null) {
                action.payload.account.uaid = createNewStateGuid();
            }
            return addAccounts(state, [action.payload.account]);

        // Remove the currently active account
        // Effect is the same as successful signout and forget
        case ActionTypes.REMOVE_ACTIVE_ACCOUNT:
            return removeAccount(state, action.payload.accountId);

        // Add the remebered accounts retrieved by the API
        case ActionTypes.REMEMBERED_ACCOUNTS_SUCCESS:
            let currentId = action.payload.currentId;
            let preserveIds = currentId ? [currentId] : undefined;
            return addAccounts(state, action.payload.accounts, preserveIds, action.payload.hasAuthenticator);

        // Remove an account that has been already signed out (if we need to forget it)
        // or updates its AuthenticatedState to signed out otherwise
        case ActionTypes.SIGNOUT_ACCOUNT_SUCCESS:
            const accountId = action.payload.accountId;
            if (action.payload.forget) {
                return removeAccount(state, accountId);
            }
            else {
                return updateAccountState(state, action.payload.accountId, {
                    authenticatedState: AuthenticatedState.NotSignedIn,
                    accountItemStatus: AccountItemStatus.Normal
                });
            }

        // Update the Account Item Status to display an error
        case ActionTypes.SIGNOUT_ACCOUNT_FAILED:
            return updateAccountState(state, action.meta.accountId, { accountItemStatus: AccountItemStatus.Error });

        // Update the Account Item Status to indicate loading while signing out in the background
        case ActionTypes.SIGNOUT_ACCOUNT_START:
            return updateAccountState(state, action.payload.accountId, { accountItemStatus: AccountItemStatus.Loading });

        // Update the state of the account when the profile picture is changed
        case ActionTypes.SET_PROFILE_PICTURE:
            return updateAccountState(state, action.payload.accountId, {
                profile: {
                    profilePictureUrl: action.payload.newUrl
                }
            });

        // Update account state with a new value for pictureUrl, and reset its load status
        case ActionTypes.SET_ACCOUNT_PICTURE:
            return updateAccountState(state, action.payload.accountId, {
                pictureUrl: action.payload.newUrl
            });

        // Update the account picture from that returned from Graph.
        case ActionTypes.GET_GRAPH_PICTURE_SUCCESS:
            return updateAccountState(state, action.payload.accountId, {
                pictureUrl: action.payload.pictureUrl
            });

        // Update all remembered account pictures that were found in cache.
        case ActionTypes.GET_CACHED_PICTURES_SUCCESS:
            let cachedResults: IGraphData[] = action.payload.cachedData;
            let newState = state;
            for (let ii = 0; ii < cachedResults.length; ii++) {
                const accountId =
                    cachedResults[ii].accountId ??
                    cachedResults[ii].payload.key;
                const account = newState[accountId];
                if (account) {
                    const { payload } = cachedResults[ii];
                    account.pictureUrl = account.pictureUrl ?? payload.resource;
                    if (account.cacheMeta) {
                        account.cacheMeta[IAccountPropTypes.PIC_URL_PROPTYPE] =
                            payload.resourceETag;
                    } else {
                        account.cacheMeta = {
                            [IAccountPropTypes.PIC_URL_PROPTYPE]:
                                payload.resourceETag
                        };
                    }
                }
            }

            return { ...newState };
        case ActionTypes.GET_CACHED_SHOW_AUTH_APP:
            let resultAuthApp: IGraphData[] = action.payload.cachedData;
            let newStateAuthApp = state;
            for (let ii = 0; ii < resultAuthApp.length; ii++) {
                const accountId =
                    resultAuthApp[ii].accountId ??
                    resultAuthApp[ii].payload.key;
                const account = newStateAuthApp[accountId];
                if (account) {
                    const { payload } = resultAuthApp[ii];
                    account.showAuthApp = payload.resource;
                }
            }

            return { ...newStateAuthApp };
        case ActionTypes.DISMISS_AUTH_APP:
            return updateAccountState(state, action.payload.accountId, {showAuthApp: false});
        default:
            return state;
    }
};

/**
 * Add an array of accounts to an Account state map
 * @param state Account state map object onto which the accounts will be added
 * @param accounts Array of accounts to add to the state
 * @param preserveIds Array of account IDs in state that will receive priority during merging (i.e.
 * will not be overriden but rather only complemented)
 * @returns A new Account state map with the added (and de-duped) accounts
 */
function addAccounts(state: IAccountStateMap, accounts: IAccount[], preserveIds?: string[], hasAuthenticator?: boolean): IAccountStateMap {
    if (accounts.length < 1) {
        return state; // avoid some unnecessary work
    }

    // Populate addedAccountsMap with AccountStates for the added accounts
    let addedAccountsMap: IAccountStateMap = {};
    for (let account of accounts) {
        const accountId = getAccountId(account);

        account[IAccountPropTypes.SHOW_AUTH_APP] = !hasAuthenticator;
        // For MSA accounts, construct the picture URL using the CID in order to download the profile image.
        if ((account.type === AccountType.MSA || account.type === AccountType.MSA_FED) && account.cid) {
            // "upck" == "user picture cache key".
            // After picture is changed, this cookie is supposed to be set for a day on *.microsoft.com. By the
            // time the cookie expires, the storage is expected to be serving updated picture.
            account.pictureUrl = account.pictureUrl;
        }

        // If the account is in the state already, we will merge its
        // values with the ones from the duplicated added account
        if (accountId in state) {
            // When adding accounts, we will give priority to new data coming in, except for
            // accounts whose IDs are in the preserveIds array. The purpose of this is,
            // particulary when merging remembered accounts, to avoid overwriting data
            // for certain accounts (namely the current one) so that we keep whatever
            // data a partner may have given us about it.
            let mergeOrder: any[] = preserveIds && preserveIds.indexOf(accountId) !== -1 ?
                [account, state[accountId]] : // Keep information from state
                [state[accountId], account]; // Override information from state

            addedAccountsMap[accountId] = merge({},
                ...mergeOrder,
                { accountItemStatus: AccountItemStatus.Normal }
            );
        }
        // Otherwise, we just add the extra fields that the state
        // requires
        else {
            addedAccountsMap[accountId] = {
                ...account,
                accountItemStatus: AccountItemStatus.Normal
            };
        }
    }

    // Merge current account map with the one for the added accounts
    return {
        ...state,
        ...addedAccountsMap
    };
}

/**
 * Remove a given account from the Account state map
 * @param accountMap Account state map object from which the account will be removed
 * @param accountId Id of the Account to remove
 * @returns A new Account state map with the passed in account removed (if it was there)
 */
function removeAccount(accountMap: IAccountStateMap, accountId: string): IAccountStateMap {
    let newAccountMap: IAccountStateMap = { ...accountMap };
    delete newAccountMap[accountId];
    return newAccountMap;
}

/**
 * Update a given account's state
 * @param state Acccount state map from which we need to update the account
 * @param accountId Id of the account to update the status of
 * @param newStatus New state of the account. Only set values are updated.
 * @returns New account state map with the updated account
 */
function updateAccountState(state: IAccountStateMap, accountId: string, newAccountState: DeepPartial<IAccountState>): IAccountStateMap {
    const account: IAccountState | undefined = state[accountId];
    if (account) {
        if (account.profile && newAccountState.profile && newAccountState.profile.profileName === '') {
            newAccountState.profile.profileName = account.profile.profileName;
        }
        return {
            ...state,
            [accountId]: {
                ...account,
                ...newAccountState as any
            }
        };
    }

    return state;
}
