import { IAccount, IOnAfterGetRememberedAccountsEventDetail, AuthenticatedState } from '@meControl/public-api';
import {
    IAuthActionCreators,
    AuthActionCreator,
    RememberedAccountActions,
    AsyncThunkAction,
    ActionTypes,
    ISignOutFromIdpArgs,
    ISignOutAndForgetFromIdpArgs,
    MeControlDispatch,
    AuthArgs
} from '@mecontrol/common';
import { SyntheticEvent, Promise, getString } from '@mecontrol/web-inline';
import { getAccountId, announce, format, getVerbalFriendlyName, getCurrentAccount, isAadLike } from '../utilities';
import { isRememberedAccountsSupported } from '../api/auth';

type MeControlAuthEventType = "beforesignin" | "beforesigninto" | "beforeswitch" | "beforeswitchto" | "beforesignoutfromapp";

const AuthOpToEventMap = {
    signIn: "beforesignin",
    signInTo: "beforesigninto",
    switch: "beforeswitch",
    switchTo: "beforeswitchto",
    signOutFromApp: "beforesignoutfromapp"
} as const;

function authActionFactory<
    AuthOp extends keyof typeof AuthOpToEventMap,
    Args extends AuthArgs[AuthOp]
>(operation: AuthOp): AuthActionCreator<Args> {
    return (args, event) => (dispatch, _, { authProvider, syntheticEventTarget }) => {
        const eventType: MeControlAuthEventType = AuthOpToEventMap[operation];
        if (eventType) {
            // one special case - beforesignin has no args at all (undefined), but the rest mirror the auth args here...
            let dispatchedEvent =
                eventType === "beforesignin"
                    ? new SyntheticEvent(eventType, undefined)
                    : new SyntheticEvent(eventType, args as any);

            if (!syntheticEventTarget.dispatchEvent(dispatchedEvent)) {
                event.preventDefault();
                return;
            }
        }


        // The result of these auth operations will come either when the webpage refreshes and re-inits
        // the MeControl with the new account state, or the app calls setActiveAccount with the
        // newly signed in account or no account if all signed out
        (authProvider[operation] as any).apply(authProvider, [args]);
    };
}

function signOutStart(dispatch: MeControlDispatch, account: IAccount, forget: boolean): void {
    dispatch({
        type: ActionTypes.SIGNOUT_ACCOUNT_START,
        payload: {
            accountId: getAccountId(account),
            forget
        }
    });
    let ariaString = account.authenticatedState == AuthenticatedState.NotSignedIn ?
        'forgetloadinga11yformat1' : 'signoutloadinga11yformat1';
    announce(format(getString(ariaString), getVerbalFriendlyName(account)));
}

function signOutSuccess(dispatch: MeControlDispatch, account: IAccount, forget: boolean): void {
    dispatch({
        type: ActionTypes.SIGNOUT_ACCOUNT_SUCCESS,
        payload: {
            accountId: getAccountId(account),
            forget
        }
    });
    let ariaString = forget ? 'signoutandforgetsuccessa11yformat1' : 'signoutsuccessa11yformat1';
    ariaString = forget && account.authenticatedState == AuthenticatedState.NotSignedIn ?
        'forgetsuccessa11yformat1' : 'signoutandforgetsuccessa11yformat1';
    announce(format(getString(ariaString), getVerbalFriendlyName(account)));
}

function signOutFailed(dispatch: MeControlDispatch, account: IAccount, forget: boolean, error: Error): void {
    dispatch({
        type: ActionTypes.SIGNOUT_ACCOUNT_FAILED,
        error: true,
        payload: error,
        meta: {
            accountId: getAccountId(account),
            forget
        }
    });
    let ariaString = account.authenticatedState == AuthenticatedState.NotSignedIn ?
        'forgeterrora11yformat1' : 'signouterrora11yformat1';
    announce(format(getString(ariaString), getVerbalFriendlyName(account)));
}

/** Fire off an async process to retrieve remembered accounts */
export function getRememberedAccounts(): AsyncThunkAction<IAccount[], RememberedAccountActions> {
    return (dispatch, getState, { authProvider, syntheticEventTarget }) => {

        // Only do any logic around rememebered accounts if they are actually
        // supported for some type of account (in which case we will trust
        // the auth provider to do the right thing)
        if (isRememberedAccountsSupported(authProvider)) {

            dispatch({
                type: ActionTypes.REMEMBERED_ACCOUNTS_START
            });

            return authProvider.getRememberedAccounts().then(accounts => {
                let currentAccount = getCurrentAccount(getState().accounts);
                let currentId = currentAccount && getAccountId(currentAccount);
                let hasAuthenticator = (currentAccount && !isAadLike(currentAccount) && currentAccount.hA) || false;

                dispatch({
                    type: ActionTypes.REMEMBERED_ACCOUNTS_SUCCESS,
                    payload: {
                        accounts,
                        currentId,
                        hasAuthenticator
                    }
                });

                let eventDetails: IOnAfterGetRememberedAccountsEventDetail = { rememberedAccounts: accounts };

                let dispatchedEvent = new SyntheticEvent("aftergetrememberedaccounts", eventDetails);
                syntheticEventTarget.dispatchEvent(dispatchedEvent);
                return accounts;
            }).catch((error: Error) => {
                dispatch({
                    type: ActionTypes.REMEMBERED_ACCOUNTS_FAILED,
                    error: true,
                    payload: error
                });

                return [];
            }).then(accounts => {
                dispatch({
                    type: ActionTypes.REMEMBERED_ACCOUNTS_COMPLETED
                })
                if (accounts && accounts.length > 0) {
                    // trigger dispatch to get remembered accounts pictures from catch
                    // dispatch(getPictureFromCache())
                }

                return accounts;
            });
        }
        else {
            dispatch({
                type: ActionTypes.REMEMBERED_ACCOUNTS_COMPLETED
            })
            return Promise.resolve([]);
        }
    };
}

export function signOutFromIdp(args: ISignOutFromIdpArgs, event: Event): AsyncThunkAction<void> {
    return (dispatch, _, { authProvider, syntheticEventTarget }) => {
        let dispatchedEvent = new SyntheticEvent("beforesignoutfromidp", args);

        if (!syntheticEventTarget.dispatchEvent(dispatchedEvent)) {
            event.preventDefault();
            return Promise.resolve();
        }

        signOutStart(dispatch, args.account, false);

        return authProvider.signOutFromIdp(args).then(() => {
            signOutSuccess(dispatch, args.account, false);
        }).catch((error: Error) => {
            signOutFailed(dispatch, args.account, false, error);
        });
    };
}

export function signOutAndForgetFromIdp(args: ISignOutAndForgetFromIdpArgs, event: Event): AsyncThunkAction<void> {
    return (dispatch, _, { authProvider, syntheticEventTarget }) => {
        let dispatchedEvent = new SyntheticEvent("beforesignoutandforgetfromidp", args);

        if (!syntheticEventTarget.dispatchEvent(dispatchedEvent)) {
            event.preventDefault();
            return Promise.resolve();
        }

        signOutStart(dispatch, args.account, true);

        return authProvider.signOutAndForgetFromIdp(args).then(() => {
            let afterEvent = new SyntheticEvent("aftersignoutandforgetfromidp", args);
            syntheticEventTarget.dispatchEvent(afterEvent);
            signOutSuccess(dispatch, args.account, true);
        }).catch((error: Error) => {
            signOutFailed(dispatch, args.account, true, error);
        });
    };
}

export const authActions: IAuthActionCreators = {
    signIn: authActionFactory("signIn"),
    signInTo: authActionFactory("signInTo"),
    switch: authActionFactory("switch"),
    switchTo: authActionFactory("switchTo"),
    signOutFromApp: authActionFactory("signOutFromApp"),
    signOutFromIdp,
    signOutAndForgetFromIdp,
    getRememberedAccounts,
};
