import {
    ActionTypes,
    AllActions,
    AsyncThunkAction,
    DataState,
    GraphResource,
    IGraphData,
    ITrackedScenarioParams,
    PictureLoadStatus,
    Scenario,
    ScenarioStage,
} from '@mecontrol/common';
import { IAccountPropTypes } from '@mecontrol/public-api';
import { logTelemetryEvent, ME } from '@mecontrol/web-inline';
import Promise from 'promise-polyfill';
import { IIFrameOperation, openIframeForCache } from '../api/iframe';
import { generateCodeChallenge } from '../utilities';

export const REMEMBERED_DONE_KEY = '__RMR_OK__';
export const CACHED_DONE_KEY = '__CACHE_OK__';
/**
 * @property {key} The hashed key to check localstorage cache
 * @property {\_timeOffSet\_} Used for checking validity of cached object
 */
export type GetCacheActions = { key: string; _timeOffSet_?: number }[];
export type SetCacheActions = (GraphResource & {
    key: string;
})[];
interface IAccountIdMapsArgs {
    mapper?: { [key: string]: string };
    key: string;
    hash: string;
    resourceType: IAccountPropTypes;
}
export function hydrateFromCache(): AsyncThunkAction<void, AllActions> {
    // Default event params set so that only a few items need to be adjusted before logging the event.
    const eventParams: ITrackedScenarioParams = {
        eventType: 'TrackedScenario',
        scenario: Scenario.Cache,
        action: '',
        previousAction: ScenarioStage.Start,
        success: true,
        durationMs: 0,
        details: ''
    };
    return (dispatch, getState) => {
        // Check remembered accounts completed
        const { picturesStatus, accounts } = getState();
        const Succeeded = PictureLoadStatus.Succeeded,
            Failed = PictureLoadStatus.Failed;
        const rmrAcctStatus = picturesStatus[REMEMBERED_DONE_KEY];
        const rmrAcctFinished =
            rmrAcctStatus &&
            (rmrAcctStatus.status === Succeeded ||
                rmrAcctStatus.status === Failed);
        if (!rmrAcctFinished) {
            return Promise.resolve();
        }
        if (!ME.Config.cache) {
            dispatch({ type: ActionTypes.GET_CACHED_DISABLED });
            return Promise.resolve();
        }
        const crypto = window.crypto || window.msCrypto;
        const allGetCacheRequests: GetCacheActions = [];
        const hashIdMapper: { [key: string]: any } = {};
        const promises = [];
        // For each account, use asynchronous crypto library to hash account ids. The thanable
        // of each promise will use helper method to push formatted obj to get-request array
        // Then execute all via Promise.all and thenable will openIframe with get-request array
        for (let key in accounts.byId) {
            promises.push(
                generateCodeChallenge(crypto, key).then(hash => {
                    createKeyAndAddToMapperAndRequestArray(
                        {
                            key,
                            mapper: hashIdMapper,
                            hash,
                            resourceType: IAccountPropTypes.PIC_URL_PROPTYPE
                        },
                        ME.Config.cacheRetention.picRetention,
                        allGetCacheRequests
                    );
                    createKeyAndAddToMapperAndRequestArray(
                        {
                            key,
                            hash,
                            mapper: hashIdMapper,
                            resourceType: IAccountPropTypes.SHOW_AUTH_APP
                        },
                        ME.Config.cacheRetention.authAppRetention,
                        allGetCacheRequests
                    );
                })
            );
        }
        return Promise.all(promises).then(values => {
            logTelemetryEvent(eventParams);
            openIframeForCache(allGetCacheRequests, undefined, getCacheOp).then(
                handleCacheHydration
            );
        });
        function handleCacheHydration(messages: IGraphData[]) {
            const pictureCacheHits = [];
            const showAuthAppCacheHits = [];
            for (let ii in messages) {
                // Reassign the unhashed accountid to cached object
                messages[ii].accountId = hashIdMapper[messages[ii].payload.key];
                if (messages[ii].status === DataState.CACHE) {
                    switch (messages[ii].payload.resourceType) {
                        case IAccountPropTypes.PIC_URL_PROPTYPE:
                            pictureCacheHits.push(messages[ii]);
                            break;
                        case IAccountPropTypes.SHOW_AUTH_APP:
                            showAuthAppCacheHits.push(messages[ii]);
                            break;
                    }
                }
            }
            if (pictureCacheHits.length > 0) {
                dispatch({
                    type: ActionTypes.GET_CACHED_PICTURES_SUCCESS,
                    payload: { cachedData: pictureCacheHits }
                });
            }
            if (showAuthAppCacheHits.length > 0) {
                dispatch({
                    type: ActionTypes.GET_CACHED_SHOW_AUTH_APP,
                    payload: { cachedData: showAuthAppCacheHits }
                });
            }
            dispatch({
                type: ActionTypes.GET_CACHED_COMPLETED
            });
        }
    };
}

export function cacheAuthAppDismissal(
    accountId: string
): AsyncThunkAction<any, AllActions> {
    return (dispatch, getState) => {
        dispatch({
            type: ActionTypes.DISMISS_AUTH_APP,
            payload: { accountId }
        });
        return generateCodeChallenge(crypto, accountId).then(hash => {
            const sets: SetCacheActions = [];
            createKeyAndAddToMapperAndRequestArray(
                {
                    key: accountId,
                    hash,
                    resourceType: IAccountPropTypes.SHOW_AUTH_APP
                },
                undefined,
                undefined,
                sets,
                false
            );
            return openIframeForCache([], sets, setAuthAppDismissalOp);
        });
    };
}
const getCacheOp: IIFrameOperation = {
    name: 'GetUsersCachedInfo',
    operation: 'GetCachedInfo',
    service: 'MeControl'
};

const setAuthAppDismissalOp: IIFrameOperation = {
    name: 'SetAuthAppDismissal',
    operation: 'SetAuthAppDismissal',
    service: 'MeControl'
};

/**
 * 
 * @param acctMeta Props will be used to generate the hashed key
 * @param timeOff If provided will determine validity of cached item
 * @param getReqs Represents the array of get requests
 * @param setReqs Represents the array of set requests
 * @param setVal The value to add to the set-item-in-cache array
 */
function createKeyAndAddToMapperAndRequestArray(
    acctMeta: IAccountIdMapsArgs,
    timeOff?: number,
    getReqs?: GetCacheActions,
    setReqs?: SetCacheActions,
    setVal?: any
) {
    const idKey = createCacheId(acctMeta.hash, acctMeta.resourceType);
    if (acctMeta.mapper && getReqs) {
        acctMeta.mapper[idKey] = acctMeta.key;
        getReqs.push({ key: idKey });
    } else if (setReqs && setVal !== undefined) {
        setReqs.push({
            key: idKey,
            resourceType: acctMeta.resourceType,
            resource: setVal
        });
    }
}
function createCacheId(hash: string, resourceType: string) {
    return `${hash}-${resourceType}`;
}
