import {
    PictureLoadStatus,
    ActionTypes,
    LoadPictureUrlActions,
    AsyncThunkAction,
    IPictureStatus,
    GetGraphPictureActions,
    Scenario,
    ScenarioStage,
    DataState,
    ISyntheticEventTarget,
    ITrackedScenarioParams,
    IAccountState,
    IGraphData,
    GraphResource
} from '@mecontrol/common';
import { IAccountPropTypes } from '@mecontrol/public-api';
import {
    Promise,
    createError,
    logTelemetryEvent,
    ME,
    SyntheticEvent,
    getOptions,
    MeControlError
} from '@mecontrol/web-inline';
import { IIFrameOperation, openIframeForProfilePictures } from '../api/iframe';
import {
    MSGraphOAuth2Builder,
    getAccountId,
    generateCodeChallenge
} from '../utilities';

export const MAX_PIC_RETRIES = 2;

/** Unique name to avoid collisions with URLS in PicStatus state obj */
export const GRAPH_PIC_STATUS_KEY = '__GRAPH_PIC_STATUS__';

export function setPictureUrlAsync(
    pictureUrl: string,
    picStatus: IPictureStatus
): AsyncThunkAction<void, LoadPictureUrlActions> {
    return (dispatch, getState) => {
        // pictureUrl is invalid -or- Picture is already loading -or- has passed its threshold of attempts to load.
        if (
            !pictureUrl ||
            picStatus.status === PictureLoadStatus.Loading ||
            picStatus.status === PictureLoadStatus.Succeeded ||
            picStatus.tries >= MAX_PIC_RETRIES
        ) {
            return Promise.resolve();
        }

        dispatch({
            type: ActionTypes.LOAD_PICTURE_URL_START,
            payload: {
                pictureUrl
            }
        });

        return downloadPicture(pictureUrl)
            .then(() => {
                dispatch({
                    type: ActionTypes.LOAD_PICTURE_URL_SUCCESS,
                    payload: {
                        pictureUrl
                    }
                });
            })
            .catch((error: Error) => {
                dispatch({
                    type: ActionTypes.LOAD_PICTURE_URL_FAILED,
                    error: true,
                    payload: error,
                    meta: {
                        pictureUrl
                    }
                });
            });
    };
}

export function downloadPicture(url?: string): Promise<void> {
    return new Promise((resolve, reject) => {
        if (!url) {
            // No picture URL, just falsify the loaded and reject the promise.
            reject(createError('Account has no picture URL'));
        } else {
            const imgSizeThreshold = 10;
            // Picture has not been loaded, try and load it.
            let img = new Image();
            let syntheticEventTarget: ISyntheticEventTarget;
            const options = getOptions();
            let imgError: MeControlError;
            if (options) {
                syntheticEventTarget = options.syntheticEventTarget;
                img.onload = () => {
                    syntheticEventTarget.dispatchEvent(
                        new SyntheticEvent('pictureupdated', {
                            completionStatus: true
                        })
                    );
                    /**
                     * Blocking Bug fix for Office SuiteHeader:
                     * https://identitydivision.visualstudio.com/Engineering/_workitems/edit/712877
                     * This is a temporary solution to be shipped for Office.
                     *
                     * Associated PBI: https://identitydivision.visualstudio.com/Engineering/_workitems/edit/720071
                     */
                    const isImgLessThanTenByTen =
                        img.width <= imgSizeThreshold &&
                        img.height <= imgSizeThreshold;
                    if (isImgLessThanTenByTen) {
                        imgError = createError(
                            `Image load error: Image is less than 10x10 pixels. This might be a transparent image!`
                        );
                        reject(imgError);
                    } else {
                        resolve();
                    }
                };
                img.onerror = ev => {
                    imgError = createError(`Failure loading URL '${url}'`);
                    syntheticEventTarget.dispatchEvent(
                        new SyntheticEvent('pictureupdated', {
                            completionStatus: false,
                            imgError
                        })
                    );
                    reject(imgError);
                };
            } else {
                reject(
                    createError(
                        `Failed to obtain MeControl options! MeControl options is null or undefined!`
                    )
                );
            }

            img.src = url;
        }
    });
}

export function getPictureFromGraph(
    currentAccount: IAccountState
): AsyncThunkAction<void, GetGraphPictureActions> {
    return (dispatch, getState) => {
        // Check and immediately resolve the promise if the configuration has disabled graph usage.
        if (!ME.Config.graphv2) {
            return Promise.resolve();
        }
        let accountId = (currentAccount && getAccountId(currentAccount)) || '';

        if (accountId === '') {
            return Promise.resolve();
        }

        // Dispatch redux START.
        dispatch({
            type: ActionTypes.GET_GRAPH_PICTURE_START,
            payload: {
                accountId
            }
        });

        // Default event params set so that only a few items need to be adjusted before logging the event.
        let eventParams: ITrackedScenarioParams = {
            eventType: 'TrackedScenario',
            scenario: Scenario.GraphPicture,
            action: '',
            previousAction: ScenarioStage.Start,
            success: true,
            durationMs: 0,
            details: ''
        };

        const crypto = window.crypto || window.msCrypto;
        const mapper: { [key: string]: any } = {};
        return generateCodeChallenge(crypto, accountId).then(
            hashedAccountId => {
                const etag = currentAccount.cacheMeta
                    ? currentAccount.cacheMeta[
                          IAccountPropTypes.PIC_URL_PROPTYPE
                      ]
                    : undefined;
                // Array of requested Graph resources
                const reqResources: GraphResource[] = [
                    {
                        key: hashedAccountId,
                        resourceType: IAccountPropTypes.PIC_URL_PROPTYPE,
                        resourceName: IAccountPropTypes.PIC_URL_PROPTYPE,
                        resourceUrl: ME.Config.graphinfo.graphphotourl,
                        resourceETag: etag,
                        resource: null
                    }
                ];
                mapper[hashedAccountId] = accountId;
                const msgraphReq = new MSGraphOAuth2Builder(
                    currentAccount,
                    hashedAccountId,
                    reqResources
                );
                return msgraphReq.initCodeFlow().then(() => {
                    // Fire the iframe to retrieve the image from graph.
                    return openIframeForProfilePictures(
                        msgraphReq,
                        getPictureOp
                    )
                        .then(frameResultResolver)
                        .catch((error: Error) => {
                            eventParams.action = DataState.FAIL;
                            eventParams.details = error.message;
                            eventParams.success = false;
                            // Disabling for now since no action/retry is attempted after dispatch
                            // dispatch({
                            //     type: ActionTypes.GET_GRAPH_PICTURE_FAILED,
                            //     error: true,
                            //     payload: error,
                            //     meta: {
                            //         accountId
                            //     }
                            // });

                            logTelemetryEvent(eventParams);
                        });
                });
            }
        );
        function frameResultResolver(message: IGraphData[]) {
            if (message !== undefined) {
                const data = message[0];
                eventParams.action = data.status;
                const resource =
                    data.payload.resource ?? data.payload.pictureUrl;
                if (data.status === DataState.OK && resource) {
                    dispatch({
                        type: ActionTypes.GET_GRAPH_PICTURE_SUCCESS,
                        payload: {
                            accountId:
                                mapper[data.payload.key ?? data.accountId],
                            pictureUrl: resource
                        }
                    });
                } else if (data?.status === DataState.NOOP) {
                    eventParams.details = 'Graph call: 304 Not Modified';
                    // dispatch({
                    //     type: ActionTypes.GET_GRAPH_PICTURE_NOOP,
                    //     payload: {
                    //         accountId: data.accountId,
                    //         data: data.payload.pictureUrl
                    //     }
                    // });
                } else {
                    eventParams.details =
                        data.error_description ??
                        data.error ??
                        'Graph call: Failed';
                    eventParams.success = false;

                    // Disabling for now since no action/retry is attempted after dispatch
                    // dispatch({
                    //     type: ActionTypes.GET_GRAPH_PICTURE_FAILED,
                    //     error: true,
                    //     payload: new Error(
                    //         `Unexpected error occurred while trying to get profile image from graph. Details: ${data}`
                    //     ),
                    //     meta: {
                    //         accountId: data.accountId
                    //     }
                    // });
                }

                logTelemetryEvent(eventParams);
            }
        }
    };
}

const getPictureOp: IIFrameOperation = {
    name: 'GetPictureFromGraph',
    operation: 'GetGraphPicture',
    service: 'MeControl'
};