import { DataState, IAuthProvider } from '@mecontrol/common';
import {
    Promise,
    perfNow,
    logTelemetryEvent,
    createError,
    w,
    getPageLocation,
    getRootDomainUrl,
    ME
} from '@mecontrol/web-inline';
import {
    OAuthFrameInitRes,
    MSGraphOAuth2Builder,
    createNewStateGuid
} from '../utilities/oauth';
import { GetCacheActions, SetCacheActions } from '../actions/cacheActions';

export interface IIFrameOperation {
    /** Name of the Me Control operation calling the IFrame */
    name: string;
    /** Name of the service or endpoint the call is going to */
    service: string;
    /** Operation name on the service/endpoint side */
    operation: string;
}
/**
 * Used for adding additional detail for logging; iframe should load and send INIT message trigger.
 */
let expectIframeResponse = { expect: false, received: false };
let iframeId = 0;
const iframeTimeout = 50 * 1000; // 50s per 99th percentile of 'me.srf' (remembered accounts) MSA Operation latency to reduce the number of false positive alerts. Source: https://aka.ms/AA4jzq2
class MeControlIframe {
    public messages: any[];
    public iframeHandle: HTMLIFrameElement;
    public iframeTargetOrigin: string;

    constructor(public src: string, expectedIframeOrigin?: string) {
        this.messages = [];
        this.iframeHandle = document.createElement('iframe');
        this.iframeHandle.id = 'mecontrol-iframe-' + iframeId++;
        (this.iframeHandle as any).sandbox =
            'allow-forms allow-scripts allow-same-origin';
        this.iframeHandle.src = src;
        this.iframeHandle.style.display = 'none';

        this.iframeTargetOrigin = expectedIframeOrigin ?? getDomainFromUrl(src);
    }

    public postMessageToIframe(
        data:
            | OAuthFrameInitRes
            | { gets?: GetCacheActions; sets?: SetCacheActions }
    ) {
        if (this.iframeHandle.contentWindow) {
            this.iframeHandle.contentWindow.postMessage(
                data,
                this.iframeTargetOrigin
            );
        }
    }
    public pushResult(data: any) {
        this.messages.push(data);
    }
}

export function openIframe(
    src: string,
    operation: IIFrameOperation,
    expectedMessageCount: number = 1,
    isExpectedMessage: (msg: any) => boolean = () => true,
    authProvider: IAuthProvider | undefined = undefined
): Promise<any[]> {
    expectIframeResponse.expect = false;
    return _openIframe(
        src,
        operation,
        expectedMessageCount,
        isExpectedMessage,
        authProvider
    );
}
export function openIframeForProfilePictures(
    msGraphReq: MSGraphOAuth2Builder,
    operation: IIFrameOperation
): Promise<any[]> {
    expectIframeResponse.expect = true;
    const iframeTargetOrigin = getDomainFromUrl(msGraphReq.redirect_uri);
    return _openIframe(
        msGraphReq.getCodeReqUrl(),
        operation,
        1,
        undefined,
        undefined,
        authHandler,
        iframeTargetOrigin
    );
    function authHandler(event: MessageEvent, meIframe: MeControlIframe) {
        const { data } = event;
        if (msGraphReq.state == data.state) {
            switch (data.status) {
                case DataState.INIT:
                    expectIframeResponse.received = true;
                    meIframe.postMessageToIframe(
                        msGraphReq.GetFrameInitResponse()
                    );
                    break;
                default:
                    meIframe.pushResult(data);
                    break;
            }
        }
    }
}
export function openIframeForCache(
    gets: GetCacheActions | undefined,
    sets: SetCacheActions | undefined,
    operation: IIFrameOperation
): Promise<any[]> {
    expectIframeResponse.expect = true;
    const rootDomain = getRootDomainUrl();
    const state = createNewStateGuid();
    const ptn = ME.Config.ptn;
    const url = `${rootDomain}/me/mecache?partner=${ptn}&wreply=${window.encodeURIComponent(
        window.location.origin
    )}`;
    const expectedMessages =
        (gets ? gets.length : 0) + (sets ? sets.length : 0);
    return _openIframe(
        url,
        operation,
        expectedMessages,
        undefined,
        undefined,
        resHandler,
        rootDomain
    );
    function resHandler(event: MessageEvent, meIframe: MeControlIframe) {
        const { data } = event;
        if (state === data.state || ptn === data.state) {
            switch (data.status) {
                case DataState.INIT:
                    expectIframeResponse.received = true;
                    meIframe.postMessageToIframe({ state, gets, sets });
                    break;
                default:
                    meIframe.pushResult(data);
                    break;
            }
        }
    }
}
/**
 *
 * @param src
 * @param operation
 * @param expectedMessageCount
 * @param isExpectedMessage
 * @param authProvider
 * @param messageHandler Tis the responsibility of this func to verify message is from correct target source
 * @param expectedIframeOrigin If none is provided, defaults to host of src arg
 * @returns
 */
export function _openIframe(
    src: string,
    operation: IIFrameOperation,
    expectedMessageCount: number = 1,
    isExpectedMessage: (msg: any) => boolean = () => true,
    authProvider: IAuthProvider | undefined = undefined,
    messageHandler?: (
        messageEvent: MessageEvent,
        meIframe: MeControlIframe
    ) => void,
    expectedIframeOrigin?: string
): Promise<any[]> {
    return new Promise((resolve, reject) => {
        if (!w.postMessage) {
            reject(
                createError('window.postMessage not supported in this browser')
            );
        }

        if (getPageLocation().protocol !== 'https:') {
            reject(
                createError(
                    'Iframes can only be opened on pages that are HTTPS secured'
                )
            );
        }

        const startTime = perfNow();
        const meIframe = new MeControlIframe(src, expectedIframeOrigin);
        let timeout = setTimeout(
            handleTimeout,
            authProvider ? 10 * 1000 : iframeTimeout
        );

        if (expectedMessageCount > 0) {
            // IE9 - addEventListener doesn't work <IE9
            w.addEventListener('message', messageHandlerWrapper);
        } else {
            meIframe.iframeHandle.addEventListener('load', () => {
                cleanup(true);
                resolve(meIframe.messages);
            });
        }

        document.body.appendChild(meIframe.iframeHandle);

        function messageHandlerWrapper(event: MessageEvent) {
            try {
                if (
                    event.origin !== meIframe.iframeTargetOrigin ||
                    meIframe.iframeHandle.contentWindow != event.source
                ) {
                    return;
                }

                if (!isExpectedMessage(event.data)) {
                    return;
                }

                // Allow custom message handling behavior
                if (!!messageHandler) messageHandler(event, meIframe);
                // default message handling
                else meIframe.messages.push(event.data);

                if (meIframe.messages.length === expectedMessageCount) {
                    cleanup(true);
                    resolve(meIframe.messages);
                } else {
                    // Waiting for more messages so reset timeout
                    clearTimeout(timeout);
                    timeout = setTimeout(handleTimeout, iframeTimeout);
                }
            } catch (error) {
                if (error instanceof Error) {
                    cleanup(
                        false,
                        error.stack ? error.stack : error.toString()
                    );
                    reject(error);
                } else {
                    const message = 'Failure in handler.';
                    cleanup(false, message);
                    reject(createError(message, false));
                }
            }
        }

        function handleTimeout(): void {
            let message = `Iframe operation to ${meIframe.iframeTargetOrigin} timed out. `;
            if (expectIframeResponse.expect && !expectIframeResponse.received) {
                message +=
                    'Iframe failed to load and send first INIT trigger to parent.';
            } else if (expectIframeResponse.expect) {
                message +=
                    'Iframe succeeded to load and send INIT trigger to parent.';
            }
            cleanup(false, message);
            reject(createError(message, true));
        }

        function cleanup(success: boolean, message?: string): void {
            clearTimeout(timeout);
            w.removeEventListener('message', messageHandlerWrapper);
            if (meIframe.iframeHandle.parentNode !== null) {
                meIframe.iframeHandle.parentNode.removeChild(
                    meIframe.iframeHandle
                );
            }

            // TODO: Can we log success here? Or should we only allow callers to log success once they have successfully
            // parsed the message and validated its response? How does that affect timing? Should we export
            // logOutgoingRequest and use it in callers of openIframe to log success of handling message? Should we have
            // multiple sub operations per one larger MeControl operation (opening iframe, parsing and handling
            // messages, the entire operation)
            try {
                logOutgoingRequest(operation, src, startTime, success, message);
            } catch {
                // Ignore logging errors
            }
        }
    });
}

/**
 * Returns the root domain for a given url
 * @param url The url to get the root domain of
 */
function getDomainFromUrl(url: string): string {
    const anchor = document.createElement('a');
    anchor.href = url;
    let domain = anchor.protocol + '//' + anchor.hostname;

    if (
        anchor.port !== null &&
        anchor.port !== '' &&
        anchor.port !== '443' &&
        anchor.port !== '80' &&
        anchor.port !== '0'
    ) {
        domain += ':' + anchor.port;
    }

    return domain;
}

/**
 * Log an OutGoingRequest event for the provided frame at the conclusion of it's call
 * @param frame Frame object containing the data to log
 * @param success Flag to indicate whether the request was successful or not
 */
function logOutgoingRequest(
    operation: IIFrameOperation,
    url: string,
    start: number,
    success: boolean,
    message?: string
): void {
    let op = operation;
    let duration = perfNow() - start;

    logTelemetryEvent({
        eventType: 'OutgoingRequest',
        serviceOperation: op.operation,
        service: op.service,
        url,
        success,
        duration,
        message
    });
}
