import { ComponentFactory, FunctionalComponent, h } from 'preact';
import { createId, ME } from '@mecontrol/web-inline';
import {
    IMeControlAccounts,
    IMeControlAppState
} from '@mecontrol/common';
import { getCurrentAccount, setQueryParams, isAadLike } from '../utilities';
import { LinkButton } from './LinkButton';
import { connect, MapDispatchToProps, MapStateToProps } from '../core/connect';
import { cacheAuthAppDismissal } from '../actions/cacheActions';

export interface IAuthAppUpsellProps {
    currentAccountid?: string;
}
export interface IAuthAppUpsellStateProps {
    accounts: IMeControlAccounts;
}
export interface IAuthAppUpsellStateDispatch {
    dismissAuthApp: (accountId: string) => void;
}
export type IAuthAppUpsellFullProps = IAuthAppUpsellProps &
    IAuthAppUpsellStateProps &
    IAuthAppUpsellStateDispatch;

const AuthAppUpsellImpl: FunctionalComponent<IAuthAppUpsellFullProps> | null =
props => {
        let uri = ME.Config.authAppUpsellUrl;
        // for now, don't display banner for non english-speaking market
        if (!uri || ME.Config.mkt.substring(0,2).toLowerCase() !== "en") {
            return null;
        }
        const { accounts, dismissAuthApp } = props;
        const currentAccount = getCurrentAccount(accounts);
        // Do not show Auth App Upsell
        if (
            !currentAccount ||
            currentAccount.showAuthApp === false ||
            isAadLike(currentAccount) || // TODO: REMOVE WHEN ENABLED FOR AAD
            (!isAadLike(currentAccount) && currentAccount.hA)
        ) {
            return null;
        }
        let banner = 'Set up Microsoft Authenticator';
        let CloseButton = (
            <div
                type="button"
                role="button"
                class="mectrl_resetStyle mectrl_authApp_close"
                onClick={() => {
                    dismissAuthApp(accounts.currentId);
                }}
            >
                <div class="mectrl_glyph glyph_authApp_close"></div>
            </div>
        );

        return (
            <div id="mectrl_authApp_banner_parent">
                <LinkButton
                    class="mectrl_authApp_banner"
                    id={createId('authApp')}
                    getUrl={() => setQueryParams(uri, {"uaid": currentAccount.uaid})}
                    openInNewTab={true}
                    uaid={currentAccount.uaid}
                >
                    <div class="mectrl_glyph glyph_authApp_icon"></div>
                    {banner}
                </LinkButton>
                {CloseButton}
            </div>
        );
    };

const mapStateToProps: MapStateToProps<
    IMeControlAppState,
    IAuthAppUpsellStateProps
> = state => ({ accounts: state.accounts });
const mapDispatchToProps: MapDispatchToProps<
    IMeControlAppState,
    IAuthAppUpsellStateDispatch
> = dispatch => ({
    dismissAuthApp(accountId): void {
        dispatch(cacheAuthAppDismissal(accountId));
    }
});
export const AuthAppUpsell: ComponentFactory<IAuthAppUpsellProps> = connect(
    mapStateToProps,
    mapDispatchToProps
)(AuthAppUpsellImpl) as any;
