import { Component } from 'preact';
import { RuntimeDisplayMode } from '@mecontrol/public-api';
import { DebounceEventHandler, debounce } from '../utilities';

export interface IBreakpointProps {
    mode: RuntimeDisplayMode;
    width: number;
    render(valueExceeded: boolean): JSX.Element;
}

export interface IBreakpointState {
    valueExceeded: boolean;
}

export class Breakpoint extends Component<IBreakpointProps, IBreakpointState> {

    private mounted: boolean;
    private watching: boolean;
    private mediaQueryWatcher: MediaQueryList | undefined;
    private readonly resizeDebouncer: DebounceEventHandler<Event>;

    constructor(props: IBreakpointProps) {
        super(props);

        // Bind functions
        this.mediaQueryHandler = this.mediaQueryHandler.bind(this);
        this.resizeDebouncer = debounce(this.resizeHandler.bind(this));

        // Initial state
        let initialExceededValue = window.innerWidth >= this.props.width;
        this.state = { valueExceeded: initialExceededValue };
        this.mounted = false;
        this.watching = false;
    }

    /** Handle resize notifications when MediaQuery is available */
    private mediaQueryHandler(mq: MediaQueryListEvent): void {
        this.breakpointCheck(mq.matches);
    }

    /** Handle debounced resize events. DO NOT USE DIRECTLY */
    private resizeHandler(evt: Event): void {
        this.breakpointCheck(window.innerWidth >= this.props.width);
    }

    /** Logic for updating state on resize events */
    private breakpointCheck(newExceedsValue: boolean): void {
        if (this.mounted && newExceedsValue != this.state.valueExceeded) {
            this.setState({ valueExceeded: newExceedsValue });
        }
    }

    /** Set up watching the size of the window */
    private setupWatcher(): void {
        if (!this.watching) {
            this.watching = true;

            // Set up watching the size of the window depending on whether
            // MediaQueries are available or not.
            if (window.matchMedia) {
                this.mediaQueryWatcher = window.matchMedia(`(min-width: ${this.props.width}px)`);
                this.mediaQueryWatcher.addListener(this.mediaQueryHandler);
            }
            else {
                window.addEventListener('resize', this.resizeDebouncer);
            }
        }
    }

    /** Stop tracking window size */
    private clearWatcher(): void {
        if (this.watching) {
            this.watching = false;

            // Clean up either media query or resize events
            if (this.mediaQueryWatcher) {
                this.mediaQueryWatcher.removeListener(this.mediaQueryHandler as any);
            }
            else {
                this.resizeDebouncer.cancel();
                window.removeEventListener('resize', this.resizeDebouncer);
            }
        }
    }

    public componentDidMount(): void {
        this.mounted = true;

        if (this.props.mode === RuntimeDisplayMode.Auto) {
            this.setupWatcher();
        }
    }

    public componentWillUnmount(): void {
        this.mounted = false;
        this.clearWatcher();
    }

    public componentDidUpdate(): void {
        if (this.props.mode === RuntimeDisplayMode.Auto) {
            this.setupWatcher();
        }
        else {
            this.clearWatcher();
        }
    }

    public render(): JSX.Element {
        // RuntimeDisplayMode.Auto === Compute 'valueExceeded'
        // RuntimeDisplayMode.Standard === Always show 'valueExceeded' experience
        // RuntimeDisplayMode.Compressed === Never show 'valueExceeded' experience
        let valueExceeded =
            (this.props.mode === RuntimeDisplayMode.Auto && this.state.valueExceeded) ||
            this.props.mode === RuntimeDisplayMode.Standard;

        return this.props.render(valueExceeded);
    }
}
