import { h, FunctionalComponent } from 'preact';
import { ICommand } from '@mecontrol/public-api';
import { IAccountState, PageAction, IAuthProvider, IAuthActions } from '@mecontrol/common';
import { createId, getString } from '@mecontrol/web-inline';
import { TruncateText, LinkButton, AuthAppUpsell } from '@mecontrol/web-boot';
import { CustomCommands } from './CustomCommands';
import { Presence } from './Presence';
import { PresenceDropdown } from './PresenceDropdown';
import { FullDisplayName } from './FullDisplayName';
import { ViewAccountLink } from './ViewAccountLink';
import { Company } from './Company';

export interface ICurrentAccountProps {
    currentAccount: IAccountState;
    authProvider: IAuthProvider;
    signOutFromApp: IAuthActions["signOutFromApp"];
    commands?: ICommand[];
}

const mainIdPart = 'currentAccount';

/**
 * Displays information about a current account such as the user's name, member name, picture, and custom commands
 * @param props The Body Props containing a current account
 */
export const CurrentAccount: FunctionalComponent<ICurrentAccountProps> = props => {
    const { currentAccount, authProvider, signOutFromApp, commands } = props;
    const pictureId = createId(mainIdPart, 'picture');
    const getSignOutFromAppUrl = () => {
        if (authProvider.navProvider) {
            return authProvider.navProvider.getSignOutFromAppUrl({ currentAccount });
        }
    };

    return (
        <div class='mectrl_currentAccount'>
            <Company account={currentAccount} />
            <div class='header_spacer'/>
            <AuthAppUpsell />
            <div class='mectrl_accountInfo'>
                <Presence {...{ currentAccount, mainIdPart, pictureId }} />
                <div class='mectrl_accountDetails'>
                    <FullDisplayName id={createId(mainIdPart, 'primary')} account={currentAccount} />
                    <TruncateText
                        id={createId(mainIdPart, 'secondary')}
                        text={currentAccount.memberName}
                    />
                    <PresenceDropdown id={'mectrl_presenceDropdown'} />
                    <ViewAccountLink account={currentAccount} />
                    <CustomCommands {...{ currentAccount, commands }} />
                </div>
            </div>
            <LinkButton
                id={createId('body', 'signOut')}
                contentId={PageAction.signOut}
                cssClass='mectrl_signout'
                getUrl={getSignOutFromAppUrl}
                ariaLabel={getString('signouta11y')}
                role={'button'}
                onClick={event => signOutFromApp({ currentAccount }, event)}
            >
                {getString('signout')}
            </LinkButton>
        </div>
    );
};
