import { h, FunctionalComponent } from 'preact';
import { IAccount, ICommand } from '@mecontrol/public-api';
import { PageAction, IValidatedMeControlConfiguration, AnchorPosition } from '@mecontrol/common';
import { getString, SyntheticEvent, createId, getOptions } from '@mecontrol/web-inline';
import { LinkButton } from '@mecontrol/web-boot';
import { Menu } from './Menu';


export interface ICustomCommandsProps {
    currentAccount: IAccount;
    commands?: ICommand[];
}

function dispatchOnClickEventHandling(event: MouseEvent, command: ICommand, currentAccount: IAccount): void {
    const commandClickEvent = new SyntheticEvent("commandclick", {
        commandId: command.id,
        currentAccount
    });

    let _meControlConfig: IValidatedMeControlConfiguration | undefined = getOptions();
    if (_meControlConfig) {
        if (!_meControlConfig.syntheticEventTarget.dispatchEvent(commandClickEvent)) {
            event.preventDefault();
        }
    }

    if (command.onClick) {
        command.onClick(commandClickEvent);
        if (commandClickEvent.defaultPrevented) {
            event.preventDefault(); // Here `event` should be the true browser event
        }
    }
}

// TODO: colejone connect component to store to clean up prop passing
export const CustomCommands: FunctionalComponent<ICustomCommandsProps> = props => {
    if (!props.commands || props.commands.length === 0) {
        return <div />;
    }

    function getUrlFactory(command: ICommand): () => string | undefined {
        return () => typeof command.url === "function"
            ? command.url({ currentAccount: props.currentAccount, commandId: command.id })
            : command.url;
    }

    const firstCommand = props.commands[0];
    const restCommands = props.commands.slice(1);

    const restCommandProps = restCommands.map((command, index) => ({
        id: createId(command.id),
        contentId: PageAction.customCommand,
        contentSlot: index + 1,
        key: command.id,
        getUrl: getUrlFactory(command),
        onClick: (event: MouseEvent) => dispatchOnClickEventHandling(event, command, props.currentAccount),
        ariaLabel: command.ariaLabel,
        children: [command.text],
        openInNewTab: command.openInNewTab
    }));
    const moreComms = getString('morecustomlinks');

    return (
        <div class='mectrl_commands'>
            <LinkButton
                id={createId(firstCommand.id)}
                contentId={PageAction.customCommand}
                contentSlot={0}
                getUrl={getUrlFactory(firstCommand)}
                onClick={(event: MouseEvent) => dispatchOnClickEventHandling(event, firstCommand, props.currentAccount)}
                cssClass='mectrl_link'
                ariaLabel={firstCommand.ariaLabel}
                openInNewTab={firstCommand.openInNewTab}
            >
                {firstCommand.text}
            </LinkButton>
            {restCommands.length > 0 ?
                <Menu
                    id={createId('moreCommands')}
                    contentId={PageAction.moreCustomCommands}
                    trigger={{
                        children: [<div class='mectrl_glyph glyph_more' />],
                        cssClass: 'mectrl_trigger',
                        ariaLabel: moreComms,
                        tooltip: true
                    }}
                    items={restCommandProps}
                    cssClass='moreCommands'
                    position={AnchorPosition.Left}
                /> :
                null
            }
        </div>
    );
};
