import { h } from "preact";
import cc from 'classcat';
import { createId } from '@mecontrol/web-inline';
import { Tooltip } from '@mecontrol/web-boot';
import { IDropdownBaseProps, DropdownBase } from './DropdownBase';

export interface IDropdownProps extends IDropdownBaseProps {
    id: string;
    renderBody(): JSX.Element;
    renderTrigger(expanded: boolean): JSX.Element;
    cssClass?: string;
    ariaLabel?: string;
    tooltip?: boolean;
}

export class Dropdown extends DropdownBase<IDropdownProps> {

    constructor(props: IDropdownProps) {
        super(props);
    }

    /** Render method for the dropdown */
    public render(): JSX.Element {
        const {
            id,
            renderBody,
            renderTrigger,
            cssClass,
            ariaLabel,
            tooltip
        } = this.props;
        const { expanded } = this.state;

        const triggerId = createId(id, 'trigger');
        const bodyId = createId(id, 'body');

        let trigger =
            <button
                id={triggerId}
                type='button'
                class='mectrl_resetStyle mectrl_trigger'
                onClick={() => this.toggleDropdown()}
                // A11y
                aria-label={ariaLabel}
                aria-expanded={expanded.toString()}
                aria-controls={bodyId}
                // Control
                data-noClose='true'
                ref={btn => this.trigger = btn as HTMLElement}
            >
                {renderTrigger(expanded)}
            </button>;

        if (tooltip && ariaLabel) {
            trigger = <Tooltip text={ariaLabel} hideOverride={expanded} >{trigger}</Tooltip>;
        }

        return (
            <div
                class={cc(['mectrl_dropdown', cssClass])}
                role={expanded ? 'dialog' : undefined}
                aria-label={expanded ? ariaLabel : undefined}
            >
                {trigger}
                <div
                    id={bodyId}
                    class={cc(['mectrl_dropdownbody', { expanded }])}
                    ref={div => this.body = div as HTMLElement}
                >
                    {renderBody()}
                </div>
            </div>
        );
    }
}
