import { Component } from "preact";
import { AnchorPosition, AnchorAlignment, ToggleAction, AnyFunction } from '@mecontrol/common';
import { traverseDomTree, shouldCloseDropdown, anchorElementTo, getFocusableElements } from '../utilities';

export interface IDropdownBaseProps {
    onToggle?(expanded: boolean, action?: ToggleAction): void;
    position?: AnchorPosition;
    alignment?: AnchorAlignment;
    defaultExpanded?: boolean;
}

export interface IDropdownBaseState {
    expanded: boolean;
}

export abstract class DropdownBase<T extends IDropdownBaseProps> extends Component<T, IDropdownBaseState> {

    protected body: HTMLElement | undefined;
    protected trigger: HTMLElement | undefined;

    private removeFocusTraps: AnyFunction | undefined;

    constructor(props: T) {
        super(props);

        this.state = { expanded: false };

        // Bind to make unique instances of methods
        this.onExpandedClick = this.onExpandedClick.bind(this);
        this.onInteractiveClick = this.onInteractiveClick.bind(this);
        this.onExpandedKeyDown = this.onExpandedKeyDown.bind(this);
        this.onExpandedResize = this.onExpandedResize.bind(this);
    }

    /**
     * Attach the appropriate event handlers on the first and last elements of the dropdown.
     * Each element will have a differently bound version of 'focusTrap' to behave
     * differently and focus on the right element.
     * A function to remove these traps will also be created and stored.
     */
    private setUpFocusTraps(): void {
        if (this.body && this.trigger) {
            // First find the first and last focusable elements
            // inside the dropdown
            let focusable = getFocusableElements(this.body);
            if (focusable.length > 0) {
                let first = this.trigger;
                let last = focusable[focusable.length - 1];

                // Create the focus trap functions via binding
                const topTrap = this.focusTrap.bind(null, true, last);
                const bottomTrap = this.focusTrap.bind(null, false, first);

                // Add traps
                first.addEventListener('keydown', topTrap);
                last.addEventListener('keydown', bottomTrap);

                // Create function to remove traps
                this.removeFocusTraps = () => {
                    first.removeEventListener('keydown', topTrap);
                    last.removeEventListener('keydown', bottomTrap);
                    this.removeFocusTraps = undefined;
                };
            }
        }
    }

    /**
     * Handles keyboard events on the first and last elements of the dropdown, with the
     * intent being to detect when focus is being taken outside of the dropdown and
     * trapping it by focusing on something else.
     * NOTE: This method is intended to be bound to the right values before being passed
     * as an event handler
     * @param shiftPressed Whether the trap should activate when the SHIFT key is pressed
     * @param focusOn What element to focus on if the trap is activated
     * @param ev The actual keyboard event
     */
    private focusTrap(shiftPressed: boolean, focusOn: HTMLElement, ev: KeyboardEvent): void {
        if (ev.key === 'Tab' && ev.shiftKey === shiftPressed) {
            ev.preventDefault();
            ev.stopPropagation();
            focusOn.focus();
        }
    }

    /**
     * Click handler to detect clicking outside of the dropdown, which causes the
     * dropdown to collapse. Used in click capturing phase
     * @param ev Click event object
     */
    private onExpandedClick(ev: MouseEvent): void {
        // If the click was outside the dropdown, we close the dropdown
        // The click will then still propagate
        const clickInTrigger = this.trigger && this.trigger.contains(ev.target as Element);
        const clickInBody = this.body && this.body.contains(ev.target as Element);
        if (!clickInBody && !clickInTrigger) {
            this.toggleDropdown(false, ToggleAction.Dismiss);
        }
    }

    /**
     * Click handler to close the dropdown as a result of an interactive element
     * click. Used in click capturing phase
     * @param ev Click event object
     */
    private onInteractiveClick(ev: MouseEvent): void {
        // Check to see if the click's target is inside of an interactive element
        let shouldClose = false;
        traverseDomTree(ev.target as Element, el => {
            if (shouldCloseDropdown(el)) {
                shouldClose = true;
            }

            // Stop traversal once we hit the root of the body, since this
            // event is only handled by the body
            return this.body && el === this.body;
        });

        if (shouldClose) {
            this.toggleDropdown(false, ToggleAction.Interaction);
        }
    }

    /**
     * Keyboard handler to detect keypresse while the dropdown is open
     * @param ev Keyboard event object
     */
    private onExpandedKeyDown(ev: KeyboardEvent): void {
        switch (ev.key) {
            // When pressing ESC, close the dropdown and stop
            // Focus is taken to the dropdown trigger
            case 'Escape':
            case 'Esc':
                ev.preventDefault();
                ev.stopPropagation();
                this.toggleDropdown(false, ToggleAction.Dismiss);
                if (this.trigger) {
                    this.trigger.focus();
                }
                break;
        }
    }

    /** Handler to close the dropdown on a window resize event */
    private onExpandedResize(): void {
        this.toggleDropdown(false, ToggleAction.Dismiss);
    }

    /**
     * Opens or closes the dropdown and handles setting / removing handlers for
     * associated interactions
     * @param expanded The new expanded value for the dropdown we are requesting.
     * Defaults to the opposite of the current state if not passed
     * @param action Optional value indicating what triggered the toggling of the dropdown
     */
    protected toggleDropdown(expanded: boolean = !this.state.expanded, action: ToggleAction = ToggleAction.Trigger): void {
        if (expanded) {
            // Position the body around the trigger based on configuration first
            if (this.body && this.trigger) {
                anchorElementTo(
                    this.body,
                    this.trigger,
                    this.props.position as AnchorPosition || AnchorPosition.Bottom,
                    this.props.alignment as AnchorAlignment || AnchorAlignment.Start
                ).then(() => {
                    // Add handlers to detect interactions outside of the control
                    document.addEventListener('click', this.onExpandedClick, true);
                    window.addEventListener('resize', this.onExpandedResize);

                    // Add handlers for interactions that close the control
                    if (this.body) {
                        this.body.addEventListener('click', this.onInteractiveClick);
                        this.body.addEventListener('keydown', this.onExpandedKeyDown);
                    }
                    if (this.trigger) {
                        this.trigger.addEventListener('keydown', this.onExpandedKeyDown);
                    }

                    // Update state and re-render
                    // We do this after positioning to avoid weird jumps
                    this.setState({ expanded: true });
                });
            }
        }
        else {
            // Remove unneeded handlers now that the dropdown is closed
            document.removeEventListener('click', this.onExpandedClick, true);
            window.removeEventListener('resize', this.onExpandedResize);
            if (this.body) {
                this.body.removeEventListener('click', this.onInteractiveClick);
                this.body.removeEventListener('keydown', this.onExpandedKeyDown);
            }
            if (this.trigger) {
                this.trigger.removeEventListener('keydown', this.onExpandedKeyDown);
            }

            // Update state and re-render
            this.setState({ expanded: false });
        }

        this.props.onToggle && this.props.onToggle(expanded, action);
    }

    public componentDidMount(): void {
        if (this.props.defaultExpanded === true) {
            this.toggleDropdown(true);
        }
    }

    /** Update method that runs after render() for the dropdown */
    public componentDidUpdate(oldProps: IDropdownBaseProps, oldState: IDropdownBaseState): void {
        const { expanded } = this.state;

        // Every time we update the component we need to update focus traps
        // in case the update changed the elements inside of it
        // To this effect, we always clear focus traps and then set them up
        // again if the dropdown is open
        this.removeFocusTraps && this.removeFocusTraps();
        if (expanded) {
            this.setUpFocusTraps();
            
            // Focus the first object within body when dropdown opens
            const justOpened = !oldState.expanded;
            if (this.body && justOpened) {
                let focusable = getFocusableElements(this.body);
                focusable[0]?.focus();
            }
        }

        // Now handle focusing on the trigger button
        // We only need to do this if we just closed the dropdown and
        // nothing else is focused
        const justClosed = !expanded && oldState.expanded;
        const focused = document.activeElement;
        const nothingFocused = focused === null || focused === undefined || focused === document.body;
        if (justClosed && nothingFocused && this.trigger) {
            this.trigger.focus();
        }
    }
}
