import { h, Component } from 'preact';
import cc from 'classcat';

export interface IFocusVisibleProps {
    cssClass: string;
    visibleOnStart?: boolean;
}

export interface IFocusVisibleState {
    activated: boolean;
}

// Class used to signal focus visible
const focusVisibleClass = 'mectrl_focus_visible';

// Number of milliseconds to wait between a keyboard and a click
// event to consider them related
const keyboardClickWaitTimeMs = 260;

export class FocusVisible extends Component<IFocusVisibleProps, IFocusVisibleState> {

    private hasRecentKeyboardAction: boolean;

    constructor(props: IFocusVisibleProps) {
        super(props);

        this.state = { activated: !!props.visibleOnStart };
        this.hasRecentKeyboardAction = false;
        this.keydownHandler = this.keydownHandler.bind(this);
        this.clickHandler = this.clickHandler.bind(this);
    }

    private toggle(activate: boolean): void {
        if (activate != this.state.activated) {
            this.setState({ activated: activate });
        }
    }

    private keydownHandler(): void {
        // Set wait time to avoid clicks triggered by keyboard to cancel this
        this.hasRecentKeyboardAction = true;
        setTimeout(() => { this.hasRecentKeyboardAction = false; }, keyboardClickWaitTimeMs);

        // Update state if necessary
        this.toggle(true);
    }

    private clickHandler(): void {
        // Toggle state when necesary, only if this click is not immediately
        // following a key press
        if (!this.hasRecentKeyboardAction) {
            this.toggle(false);
        }
    }

    public componentDidMount(): void {
        document.addEventListener('keydown', this.keydownHandler, true);
        document.addEventListener('click', this.clickHandler, true);
    }

    public componentWillUnmount(): void {
        document.removeEventListener('keydown', this.keydownHandler, true);
        document.removeEventListener('click', this.clickHandler, true);
    }

    public render(): JSX.Element {
        const { cssClass, children } = this.props;
        const { activated } = this.state;

        return (
            <div class={cc([cssClass, { [focusVisibleClass]: activated }])}>
                {children}
            </div>
        );
    }
}
