import { h, FunctionalComponent } from 'preact';
import { AccountType } from '@mecontrol/public-api';
import { IAccountState, PageAction } from '@mecontrol/common';
import { getString, createId } from '@mecontrol/web-inline';
import { LinkButton, getFullDisplayName, TruncateText, raiseSyntheticEvent } from '@mecontrol/web-boot';

export interface IFullDisplayNameProps {
    id: string;
    account: IAccountState;
}

/**
 * Shows the full display name of the user, or if the user has no name, a link to add a name.
 * @param props The Body Props containing a current account
 */
export const FullDisplayName: FunctionalComponent<IFullDisplayNameProps> = props => {
    const fullDisplayName = getFullDisplayName(props.account);
    if (fullDisplayName !== '') {
        return (
            <TruncateText
                id={props.id}
                cssClass='mectrl_name'
                text={fullDisplayName}
                ariaLabel={fullDisplayName}
                hideTooltip={false}
                showToolTipOnClick={true}
            />
        );
    }

    if (props.account.type == AccountType.AAD) {
        return null;
    }
    else {
        // If full name didn't return or if there was no current account, show an add name link
        // Note: This should never happen for AAD accounts
        let uri = "https://account.microsoft.com/profile/edit-name?ref=MeControl";
        return (
            <LinkButton
                id={createId('addName')}
                contentId={PageAction.addName}
                getUrl={() => uri}
                onClick={mouseEvent => raiseSyntheticEvent('uriClickEvent', { uri }, mouseEvent)}
                cssClass='mectrl_link'
                ariaLabel={getString('addyournamelabel')}
                openInNewTab={true}
            >
                {getString('addyourname')}
            </LinkButton>
        );
    }
};
