import { h, FunctionalComponent, ComponentFactory } from "preact";
import cc from 'classcat';
import { IAccountState, IMeControlAppState } from '@mecontrol/common';
import { createId } from '@mecontrol/web-inline';
import { ProfilePicture } from './ProfilePicture';
import { TruncateText } from './TruncateText';
import { getHeaderDisplayName, isSignedIn, getThemeClass } from '../utilities';
import { PresenceAsync } from './PresenceAsync';
import { ITheme } from '@mecontrol/public-api';
import { MapStateToProps, MapDispatchToProps, connect } from '../core/connect';

export interface IHeaderProps {
    text?: string;

    account?: IAccountState;
    hideText?: boolean;
    hideIcon?: boolean;
}

interface IHeaderStateProps {
    headerTheme?: ITheme;
}

interface IHeaderDispatchProps { }

type IHeaderFullProps = IHeaderProps & IHeaderStateProps & IHeaderDispatchProps;

const headerClass = "mectrl_topHeader";

const HeaderImpl: FunctionalComponent<IHeaderFullProps> = props => {
    const { account, hideIcon } = props;
    const signedIn = account && isSignedIn(account);
    const displayText = signedIn ? getHeaderDisplayName(account) : props.text;
    let cssClass = headerClass;
    if (props.headerTheme) {
        cssClass = cc([headerClass, getThemeClass(props.headerTheme)]);
    }
    const text = props.hideText || !displayText ? null :
        <TruncateText text={displayText} cssClass={cc(['mectrl_header_text', { noIcon: hideIcon }])} hideTooltip={true} />;

    const icon = hideIcon ? null :
        [
            <ProfilePicture id={createId('headerPicture')} currentAccount={account} graphCall={true} />,
            signedIn ? <PresenceAsync /> : null
        ];

    return (
        <div class={cssClass} aria-hidden={true} role='presentation'>
            {text}
            {icon}
        </div>
    );
};

const mapStateToProps: MapStateToProps<IMeControlAppState, IHeaderStateProps> = state => ({
    headerTheme: state.headerTheme
});

const mapDispatchToProps: MapDispatchToProps<IMeControlAppState, IHeaderDispatchProps> = () => ({});

export const Header: ComponentFactory<IHeaderProps> = connect(mapStateToProps, mapDispatchToProps)(HeaderImpl);
