import { h, FunctionalComponent } from 'preact';
import cc from 'classcat';
import { ContentSource } from '@mecontrol/common';
import { logTelemetryEvent } from '@mecontrol/web-inline';
import { Tooltip } from './Tooltip';
import { TruncateTag } from './TruncateTag';
import { toChildArray } from '../core/toChildArray';

export interface ILinkButtonProps {
    id: string;
    contentId?: string;
    contentType?: string;
    contentSlot?: number;
    getUrl?(): string | undefined;
    onClick?: JSX.MouseEventHandler;
    isDisabled?: boolean;
    cssClass?: string;
    ariaLabel?: string;
    openInNewTab?: boolean;
    [key: string]: any;
}

export const LinkButton: FunctionalComponent<ILinkButtonProps> = props => {
    // Yarg! There be no reason ter check for missing props!
    const {
        id,
        contentId,
        contentType,
        contentSlot,
        isDisabled = false,
        getUrl,
        onClick,
        cssClass,
        ariaLabel,
        children,
        accountType,
        openInNewTab,
        uaid,
        ...otherAttributes
    } = props;
    const classes = cc(['mectrl_resetStyle', cssClass]);
    const url = getUrl && getUrl();

    const handleClick = (ev: MouseEvent) => {
        logTelemetryEvent({
            eventType: 'PageAction',
            content: {
                id: contentId || id,
                type: contentType,
                slot: contentSlot,
                source: ContentSource.Action
            },
            targetUrl: url,
            uaid: uaid,
        });
        onClick && onClick(ev);
    };

    // Render a button when disabled or when we have no URL to link to
    let tag: any = isDisabled || !url ? 'button' : 'a';

    // Basic attributes that button or a tags will share
    let attributes: any = {
        id,
        class: classes,
        ['aria-label']: ariaLabel,
        ...otherAttributes
    };

    // Add button type to avoid form submission
    if (tag === 'button') {
        attributes.type = 'button';
    }

    // Add special case attributes
    if (isDisabled) {
        attributes.disabled = true;
    }
    else {
        attributes.onClick = handleClick;
        // For a tag
        if (url) {
            attributes.href = url;
            attributes.target = openInNewTab ? '_blank' : '_top';
        }
    }

    let childArray = toChildArray(children);

    // If this LinkButton has only a string as its child, then we will truncate
    // said string by actually rendering a TruncateTag component
    if (childArray && childArray.length == 1 && typeof childArray[0] === 'string') {
        attributes.tag = tag;
        attributes.title = attributes.title || childArray[0];
        tag = TruncateTag;
    }
    // For other cases where we have a title attribute, we will wrap our tag in a
    // Tooltip component to render that correctly
    else if (attributes.title) {
        let title = attributes.title;
        delete attributes.title;

        childArray = [h(tag, attributes, childArray || null)];
        attributes = { text: title };
        tag = Tooltip;
    }

    return h(tag, attributes, childArray || null);
};
