import { h, Component } from 'preact';
import cc from 'classcat';
import { AnchorAlignment, AnchorPosition, ContentSource } from '@mecontrol/common';
import { createId, logTelemetryEvent } from '@mecontrol/web-inline';
import { ILinkButtonProps, DebounceEventHandler, Tooltip, LinkButton, debounce } from '@mecontrol/web-boot';
import { anchorElementTo, getFocusableElements } from '../utilities';

// Get the main div that contains all of the Body component via selector
const getMeControlBody = () => {
    let body = document.querySelector('.mectrl_body');
    return body ? body as HTMLElement : undefined;
};

export type IMenuItemProps = ILinkButtonProps & { key: string };
export interface IMenuTriggerProps {
    children: JSX.Element | JSX.Element[];
    cssClass?: string;
    ariaLabel?: string;
    tooltip?: boolean;
}

export interface IMenuProps {
    id: string;
    contentId?: string;
    contentSlot?: number;
    trigger: IMenuTriggerProps;
    items: IMenuItemProps[];
    cssClass?: string;
    position?: AnchorPosition;
    alignment?: AnchorAlignment;
}

export interface IMenuState {
    expanded: boolean;
}

declare const enum FocusChange {
    Forward,
    Backward,
    Start,
    End
}

/** Class used to tell the Menu to use fixed-positioning */
export const fixedMenuClass = 'fixed-menu';

/**
 * Menu component with keyboard interaction.
 *
 * For more details on expected behavior and examples, go to the WAI ARIA practices site here:
 * https://www.w3.org/TR/wai-aria-practices/#menubutton
 *
 * Example implementation:
 * https://www.w3.org/TR/wai-aria-practices/examples/menu-button/menu-button-links.html
 */
export class Menu extends Component<IMenuProps, IMenuState> {

    private menuRoot: HTMLElement | undefined;
    private trigger: HTMLElement | undefined;
    private menu: HTMLElement | undefined;

    private readonly scrollHandler: DebounceEventHandler<Event>;

    private focusIndex: number = -1;
    private focusAtStart: boolean = true;

    constructor(props: IMenuProps) {
        super(props);

        this.expandedClickHandler = this.expandedClickHandler.bind(this);
        this.expandedResizeHandler = this.expandedResizeHandler.bind(this);
        this.menuButtonKeyboardHandler = this.menuButtonKeyboardHandler.bind(this);
        this.menuItemKeyboardHandler = this.menuItemKeyboardHandler.bind(this);
        this.scrollHandler = debounce(this.expandedScrollHandler.bind(this));

        this.state = { expanded: false };
    }

    /**
     * Toggle menu to open/closed configurations and update associated
     * event handlers
     * @param expanded New state for the menu
     */
    private toggleMenu(expanded: boolean = !this.state.expanded): void {
        if (expanded) {

            // Position the body around the trigger based on configuration
            if (this.menu && this.trigger) {
                anchorElementTo(
                    this.menu,
                    this.trigger,
                    this.props.position || AnchorPosition.Bottom,
                    this.props.alignment || AnchorAlignment.Start,
                    getMeControlBody(),
                    true
                ).then(() => {
                    // Set up event listeners
                    document.addEventListener('click', this.expandedClickHandler, true);
                    document.addEventListener('scroll', this.scrollHandler, true);
                    window.addEventListener('resize', this.expandedResizeHandler);

                    // Update state and re-render
                    // Do this after anchoring to avoid weird jumps in position
                    this.setState({ expanded: true });
                });
            }

            // Fire telemetry for opening the menu
            logTelemetryEvent({
                eventType: 'PageAction',
                isCritical: false,
                content: {
                    id: this.props.contentId || this.props.id,
                    slot: this.props.contentSlot,
                    source: ContentSource.Action
                }
            });
        }
        else {
            // Remove listeners
            document.removeEventListener('click', this.expandedClickHandler, true);
            document.removeEventListener('scroll', this.scrollHandler, true);
            window.removeEventListener('resize', this.expandedResizeHandler);

            // Update state and re-render
            this.setState({ expanded: false });
        }
    }

    /**
     * Move focus to the menu's children based on the passed in parameter
     * @param change Parameter that helps determine where to move focus to
     */
    private moveFocus(change: FocusChange): void {
        if (this.state.expanded && this.menu) {
            const menuItems = getFocusableElements(this.menu, true);
            const totalItems = menuItems.length;

            if (totalItems > 0) {

                switch (change) {
                    case FocusChange.Forward:
                        this.focusIndex = (this.focusIndex + 1) % totalItems;
                        break;
                    case FocusChange.Backward:
                        this.focusIndex = (this.focusIndex - 1 + totalItems) % totalItems;
                        break;
                    case FocusChange.Start:
                        this.focusIndex = 0;
                        break;
                    case FocusChange.End:
                        this.focusIndex = totalItems - 1;
                        break;
                }

                menuItems[this.focusIndex].focus();
            }
        }
    }

    /**
     * Click handler to detect click interactions with the menu.
     * Primarily focused on closing the menu when either clicking outside
     * of it or when clickin a link inside of it.
     * @param ev Click event object
     */
    private expandedClickHandler(ev: MouseEvent): void {
        // If the click was outside the menu, we close it
        if (!this.menuRoot || !this.menuRoot.contains(ev.target as Element)) {
            this.toggleMenu(false);

            // Now we check if the click was inside the body, in which case
            // we stop the event from propagating
            // Clicking outside the MeControl will not stop propagation
            const meControlBody = getMeControlBody();
            if (meControlBody && meControlBody.contains(ev.target as Element)) {
                ev.preventDefault();
                ev.stopPropagation();
            }
        }
        // Otherwise, we verify if any of the children were clicked, which
        // equates to an interactive click, which should close the menu too
        else if (this.menu) {
            const menuItems = getFocusableElements(this.menu, true);
            if (menuItems.some(item => item.contains(ev.target as HTMLElement))) {
                this.toggleMenu(false);
            }
        }
    }

    /** Handler for debounced scroll events. DO NOT USE DIRECTLY */
    private expandedScrollHandler(event: Event): void {
        // Only use this logic if there is a root and event target
        // to use. Otherwise, just collapse the menu
        if (this.menuRoot && event.target) {
            // If the scroll event happened inside the menu, we won't
            // close the menu
            if (!this.menuRoot.contains(event.target as any)) {
                this.toggleMenu(false);
            }
        }
        else {
            this.toggleMenu(false);
        }
    }

    /** Handler to close the menu on any resize event */
    private expandedResizeHandler(): void {
        this.toggleMenu(false);
    }

    /**
     * Handle keyboard interactions when focus is on the menu trigger
     * @param ev Keyboard event
     */
    private menuButtonKeyboardHandler(ev: KeyboardEvent): void {
        this.focusAtStart = true;
        let allowPropagation = false;
        switch (ev.key) {
            case 'Up': // IE and older Firefox
            case 'ArrowUp':
                this.focusAtStart = false;
                this.toggleMenu(true);
                break;

            case 'Down': // IE and older Firefox
            case 'ArrowDown':
                this.toggleMenu(true);
                break;

            // Allow other keys to pass through
            default:
                allowPropagation = true;
                break;
        }

        if (!allowPropagation) {
            ev.preventDefault();
            ev.stopPropagation();
        }
    }

    /**
     * Handle keyboard interactions when focus is on the menu itself
     * @param ev Keyboard event
     */
    private menuItemKeyboardHandler(ev: KeyboardEvent): void {
        let allowPropagation = false;
        switch (ev.key) {
            case 'Esc': // Old IE and Firefox versions
            case 'Escape':
                this.toggleMenu(false);
                break;

            case 'Up': // IE and older Firefox
            case 'ArrowUp':
                this.moveFocus(FocusChange.Backward);
                break;

            case 'Down': // IE and older Firefox
            case 'ArrowDown':
                this.moveFocus(FocusChange.Forward);
                break;

            case 'Home':
                this.moveFocus(FocusChange.Start);
                break;

            case 'End':
                this.moveFocus(FocusChange.End);
                break;

            case 'Tab':
                this.toggleMenu(false);

            // Allow other keys to pass through
            default:
                allowPropagation = true;
                break;
        }

        if (!allowPropagation) {
            ev.preventDefault();
            ev.stopPropagation();
        }
    }

    /** Update method that runs after render() for the menu */
    public componentDidUpdate(oldProps: IMenuProps, oldState: IMenuState): void {
        const { expanded } = this.state;

        // Ensure this only happens the first time we update after a state change
        if (oldState.expanded !== expanded) {
            if (expanded) {
                this.moveFocus(this.focusAtStart ? FocusChange.Start : FocusChange.End);
            }
            else {
                this.trigger && this.trigger.focus();
            }
        }
    }

    /** Render method for the menu */
    public render(): JSX.Element {
        const {
            id,
            trigger,
            items,
            cssClass,
        } = this.props;
        const { expanded } = this.state;

        const triggerId = createId(id, 'trigger');
        const menuId = createId(id, 'menu');

        let triggerBtn =
            <button
                // Standard button attr
                id={triggerId}
                type='button'
                class={cc(['mectrl_resetStyle', trigger.cssClass, { expanded }])}
                onClick={() => this.toggleMenu()}
                // A11y
                aria-label={trigger.ariaLabel}
                aria-haspopup='true'
                aria-controls={menuId}
                aria-expanded={expanded.toString()}
                data-noClose='true'
                // Menu control
                tabIndex={expanded ? -1 : 0}
                ref={btn => this.trigger = btn as HTMLElement}
                onKeyDown={this.menuButtonKeyboardHandler}
            >
                {trigger.children}
            </button>;

        if (trigger.ariaLabel && trigger.tooltip) {
            triggerBtn = <Tooltip text={trigger.ariaLabel}>{triggerBtn}</Tooltip>;
        }

        return (
            <div
                class={cc(['mectrl_menu', cssClass])}
                ref={div => this.menuRoot = div as HTMLElement}
            >
                {triggerBtn}
                <ul
                    id={menuId}
                    // A11y
                    role='menu'
                    aria-labelledby={triggerId}
                    // Menu control
                    class={expanded ? 'expanded' : ''}
                    ref={mul => this.menu = mul as HTMLElement}
                    onKeyDown={this.menuItemKeyboardHandler}
                >
                    {items.map((itemProps, index) => (
                        <li key={itemProps.key} role='none'>
                            <LinkButton
                                // Item props
                                {...itemProps}
                                // Control
                                tabIndex={-1}
                                // A11y
                                role='menuitem'
                                aria-posinset={index + 1}
                                aria-setsize={items.length}
                            />
                        </li>
                    ))}
                </ul>
            </div>
        );
    }
}
