import { h, FunctionalComponent, ComponentFactory } from 'preact';
import { IPresenceState, IPresenceConfig } from '@mecontrol/public-api';
import { AnchorPosition, PageAction, DispatchWithPresenceState, IMeControlAppState } from '@mecontrol/common';
import { createId } from '@mecontrol/web-inline';
import { MapStateToProps, MapDispatchToProps, connect, setPresenceToPartner } from '@mecontrol/web-boot';
import { Menu } from './Menu';

export interface IPresenceDropdownOwnProps {
    id: string;
}

interface IPresenceDropdownDispatchProps {
    setPresenceToPartner?: DispatchWithPresenceState;
}

interface IPresenceDropdownStateProps {
    presenceConfig?: IPresenceConfig;
}

export type IPresenceDropdownProps = IPresenceDropdownOwnProps & IPresenceDropdownDispatchProps & IPresenceDropdownStateProps;

const PresenceDropdownImpl: FunctionalComponent<IPresenceDropdownProps> = props => {
    let presenceDropdown = null;
    let { id, presenceConfig } = props;

    if (presenceConfig && presenceConfig.currentPresence && presenceConfig.presenceList) {
        const presenceListStates = presenceConfig.presenceList.map((presenceState: IPresenceState, index: number) => ({
            id: createId(presenceState.presenceString),
            contentId: PageAction.presenceStateChange,
            contentSlot: index,
            key: presenceState.presenceString,
            getUrl: undefined,
            onClick: () => {
                if (props.setPresenceToPartner) {
                    props.setPresenceToPartner(presenceState);
                }
            },
            ariaLabel: presenceState.presenceString,
            children: [
                <div class="mectrl_presence_indicator_container">
                    <div style={{ backgroundImage: 'url(' + presenceState.imageUrl + ')' }}></div>
                </div>,
                <div class="mectrl_presence_string_container">{presenceState.presenceString}</div>
            ],
            openInNewTab: false,
            ['data-noClose']: 'true'
        }));

        const currentPresence = presenceConfig.currentPresence.presenceString;
        presenceDropdown =
            <Menu
                id={id}
                contentId={PageAction.presenceDropdown}
                trigger={{
                    children: [
                        <div class="mectrl_presence_dropdown_text">{currentPresence}</div>,
                        <div class="mectrl_presence_dropdown_glyph glyph_chevron"></div>
                    ],
                    cssClass: 'mectrl_trigger',
                    ariaLabel: currentPresence
                }}
                items={presenceListStates}
                cssClass='presenceDropdown'
                position={AnchorPosition.Bottom}
            />;
    }

    return (presenceDropdown);
};

const mapStateToProps: MapStateToProps<IMeControlAppState, IPresenceDropdownStateProps> = state => ({
    presenceConfig: state.presenceConfig
});

export const mapDispatchToProps: MapDispatchToProps<IMeControlAppState, IPresenceDropdownDispatchProps> = dispatch => {
    return {
        setPresenceToPartner(presenceState: IPresenceState): void {
            dispatch(setPresenceToPartner(presenceState));
        }
    };
};

export const PresenceDropdown: ComponentFactory<IPresenceDropdownOwnProps> = connect(mapStateToProps, mapDispatchToProps)(PresenceDropdownImpl);
