import { h, Component, ComponentFactory } from 'preact';
import cc from 'classcat';
import {
    IAccountState,
    PictureLoadStatus,
    IMeControlAppState,
    IPictureStatusDictionary,
    IPictureStatus
} from '@mecontrol/common';
import { connect, MapDispatchToProps, MapStateToProps } from '../core/connect';
import {
    getPictureFromGraph,
    GRAPH_PIC_STATUS_KEY,
    MAX_PIC_RETRIES,
    setPictureUrlAsync
} from '../actions/pictureActions';
import { getPictureUrl, delayUntilAnimationFrame } from '../utilities';
import { CACHED_DONE_KEY } from '../actions/cacheActions';
import { IAccountPropTypes } from '@mecontrol/public-api';

export interface IProfilePictureProps {
    id?: string;
    currentAccount?: IAccountState;
    glyphClassOverride?: string;
    /** Indicate whether graph call could be made on behalf of this profilepic tile. */
    graphCall?: boolean;
}

export interface IProfilePictureStateProps {
    picturesStatus: IPictureStatusDictionary;
}

export interface IProfilePictureDispatchProps {
    loadPicture: (pictureUrl: string, picStatus: IPictureStatus) => void;
    fetchMSGraphPicture: (currentAccount: IAccountState) => void;
}

export type IProfilePictureFullProps = IProfilePictureProps &
    IProfilePictureStateProps &
    IProfilePictureDispatchProps;

/**
 * Constant value change for decriminting the font size to ensure it will fit within the container bounds. The higher the
 * number, the less loops (calculations) that will need to take place to determine the best fit.  but too big of a number
 * will result in items being smaller than they need to be.
 */
const fontShrinkDelta: number = 2;

/**
 * Constant value for the min number of characters that initials can consist of in order to be considered valid. This has
 * been added as a range in order to support lightweight accounts that may only have a single name which will result in
 * a single letter initial.
 */
const initialsMinLength: number = 1;

/**
 * Constant value for the max number of characters that initials can consist of in order to be considered valid. This has
 * been added as a range to deal with accounts that are light weight as well as standard accounts that may have multiple names.
 * by having a max, we can ensure consistant experiences for user.
 */
const initialsMaxLength: number = 2;

/**
 * Implementation of the Profile Picture component. This component will logically determine what to show within the confindes
 * of the picture area. Meaning if no picture is present to be shown, an appropriate fallback will happen showing either a
 * glyph appropriate for the account type, or the initials of the account if they were provided within the current account details.
 * Additionally, this component will ensure the profile picture does in fact load, and if not, will revert back to the fallback
 * instead of showing a failed to load image.
 */
export class ProfilePictureImpl extends Component<IProfilePictureFullProps> {
    /**
     * In memory canvas that can be used to measure the initials to ensure they don't overflow. This field
     * will be left as undefined unless it is needed for measuring.
     */
    private localCanvas?: HTMLCanvasElement;

    /**
     * Keeps a cache of computed sizes for initials so that during re-renders, we do not have to re-compute.
     */
    private computedSizesCache: Record<string, string> = {};

    /**
     * Tracks whether the computation is currently running when determining the sites of the initials.
     */
    private isComputingInitials: boolean = false;

    /**
     * Constructs a new instance of the ProfilePictureImpl component.
     * @param props The full set of properties, state, and dispatch for the profile picture component.
     */
    constructor(props: IProfilePictureFullProps) {
        super(props);
    }

    /**
     * Renders the component into a JSX element.
     */
    public render(): JSX.Element {
        let {
            id,
            currentAccount,
            glyphClassOverride,
            graphCall,
            loadPicture,
            fetchMSGraphPicture,
            picturesStatus
        } = this.props;
        let inlineStyles: { backgroundImage?: string; fontSize?: string } = {};
        let cssClasses = ['mectrl_profilepic'];
        let pictureUrl = getPictureUrl(currentAccount);
        let picStatus = this.getPictureStatus(pictureUrl, picturesStatus);
        let initials: string | undefined;

        if (picStatus.status === PictureLoadStatus.Succeeded) {
            inlineStyles.backgroundImage = `url('${pictureUrl}')`;
        } else {
            // Kick off async dispatches for pictures
            // Use PicLoadStatus to check if getRememberedAccounts finished
            const Succeeded = PictureLoadStatus.Succeeded,
                Failed = PictureLoadStatus.Failed;
            const cacheStatus = this.getPictureStatus(
                CACHED_DONE_KEY,
                picturesStatus
            ).status;
            const cacheFinished = cacheStatus === Succeeded;
            const foundCachedPic =
                currentAccount &&
                currentAccount.cacheMeta &&
                currentAccount.cacheMeta[IAccountPropTypes.PIC_URL_PROPTYPE];
            if (
                graphCall &&
                currentAccount &&
                cacheFinished &&
                (!pictureUrl ||
                    (picStatus.tries === MAX_PIC_RETRIES &&
                        picStatus.status === Failed) ||
                    foundCachedPic)
            ) {
                // Check if main pic tile and currentAccount and no url provided or url failed
                // After max retries of url failure, attempt graph call
                let msGraphPicStatus = this.getPictureStatus(
                    GRAPH_PIC_STATUS_KEY,
                    picturesStatus
                );
                // Ensures only fires off fetch graph pic event once
                if (msGraphPicStatus.status === PictureLoadStatus.None) {
                    fetchMSGraphPicture(currentAccount);
                }
            } else {
                /*
                 * This component will trigger the promise to try and load the image every time it refrehes
                 * to ensure the image does get loaded.  The dispatch method sent down will have a retry count to prevent
                 * over use of the calls after a certain number of failures. Because the users image site is unreliable, having
                 * the retries each re-render will help clean up when images don't load on the first try.
                 */
                loadPicture(pictureUrl, picStatus);
            }
            // Check if we need to render initials
            if (this.shouldRenderInitials(currentAccount)) {
                // Adding typescript certainty bypass since shouldRenderinitials helper method already checks existance
                initials = currentAccount!.initials!;
                cssClasses.push('mectrl_profilepic_initials');

                // If undefined, ensure to set as blank to wipe any any previous fontSize value. if left undefined, it will merge with previous.
                inlineStyles.fontSize = this.computedSizesCache[initials] || '';
            } // or just a pawn/glyph
            else {
                cssClasses.push(
                    'mectrl_glyph',
                    this.getValidGlyph(glyphClassOverride, currentAccount)
                );
            }
        }

        return (
            <div
                id={id}
                class={cc(cssClasses)}
                style={inlineStyles}
                aria-hidden={true}
                role="presentation"
            >
                {initials}
            </div>
        );
    }

    /**
     * Triggered method for when the component successfully loaded/mounted.
     */
    public componentDidMount(): void {
        this.triggerOverflowCalculation();
    }

    /**
     * Triggered method for when the component successfully updates.
     */
    public componentDidUpdate(previousProps: IProfilePictureFullProps): void {
        this.triggerOverflowCalculation();
    }

    /**
     * Helper to trigger the overflow calulation if conditions are met.
     */
    private triggerOverflowCalculation(): void {
        if (!this.isComputingInitials) {
            this.isComputingInitials = true;
            delayUntilAnimationFrame(() => {
                if (
                    this.base &&
                    this.base.innerHTML &&
                    this.computedSizesCache[this.base.innerHTML] == undefined
                ) {
                    this.fitOverflowInitialsWithCanvas(this.base);
                }

                this.isComputingInitials = false;
            });
        }
    }

    /**
     * Helper method that will measure the inner text (initials) of the current account against the size
     * of the container's client width. If an overflow of text will occur, this method will apply an
     * inline style override for the font-size to ensure the text is visible properly.
     * @param container The containing HTML Element that contains the text being measured and adjusted.
     */
    private fitOverflowInitialsWithCanvas(container: HTMLElement): void {
        // Instantiate the local (in memory) canvas for the class if undefined
        if (this.localCanvas === undefined) {
            this.localCanvas = document.createElement('canvas');
        }

        // Initialize and compute values needed to run measureText tests on canvas context.
        const computedStyle: CSSStyleDeclaration = getComputedStyle(
            container,
            null
        );
        const fontFamily = computedStyle.getPropertyValue('font-family');
        const originalFontSize = computedStyle.getPropertyValue('font-size');
        const originalSize = parseFloat(originalFontSize);
        let canvasContext = this.localCanvas.getContext(
            '2d'
        ) as CanvasRenderingContext2D;

        // Start with initial values
        let newSize = originalSize;
        canvasContext.font = `${newSize}px ${fontFamily}`;
        let measuredTextWidth: number = canvasContext.measureText(
            container.innerHTML
        ).width;
        let loopCount = 0;
        const maxLoops = 10;

        // If values cause overflow, being loop to find the right font size to allow text to not overflow
        while (
            measuredTextWidth > container.clientWidth &&
            loopCount++ < maxLoops
        ) {
            newSize -= fontShrinkDelta;
            canvasContext.font = `${newSize}px ${fontFamily}`;
            measuredTextWidth = canvasContext.measureText(
                container.innerHTML
            ).width;
        }

        // Check and update if size actually changed due to loop above or not
        if (originalSize != newSize) {
            container.style.fontSize = `${newSize}px`;
            this.computedSizesCache[container.innerHTML] =
                container.style.fontSize;
        } else {
            // set computed size to empty to represent the "default".
            this.computedSizesCache[container.innerHTML] = '';
        }
    }

    /**
     * Simple helper method to get the status of the picture from the state. If no state is found a new
     * initial state will be created and returned to the caller.
     * @param pictureUrl The string URL for the picture to retrieve state on.
     * @param picturesStatus The state of IPictureStatusDictionary.
     */
    private getPictureStatus(
        pictureUrl: string,
        picturesStatus: IPictureStatusDictionary
    ): IPictureStatus {
        return (
            picturesStatus[pictureUrl] || {
                status: PictureLoadStatus.None,
                tries: 0
            }
        );
    }

    /**
     * Simple helper method that will determine which glyph class or otherwise to use for this picture
     * component based on the current account's information and state.
     * @param glyphClass Any optional overriding glyph class that was requested by the props.
     * @param currentAccount The current account state associated with this profile picture.
     */
    private getValidGlyph(
        glyphClass: string | undefined,
        currentAccount: IAccountState | undefined
    ): string {
        if (!glyphClass) {
            if (!currentAccount) {
                return 'mectrl_signIn_circle_glyph';
            } else {
                // There are three possible types - "aad", "msa" or "msaFed".
                // The last two are both MSAs, with one coming from MSA IDP and one coming from AAD IDP.
                // From a glyph perspective, both "msa" and "msaFed" are personal accounts.
                return currentAccount.type === 'aad'
                    ? 'glyph_aadAccount_circle'
                    : 'glyph_account_circle';
            }
        }

        return glyphClass;
    }
    private shouldRenderInitials(
        currentAccount: IAccountState | undefined
    ): boolean {
        return (
            !!currentAccount &&
            !!currentAccount.initials &&
            currentAccount.initials.length >= initialsMinLength &&
            currentAccount.initials.length <= initialsMaxLength
        );
    }
}

const mapStateToProps: MapStateToProps<
    IMeControlAppState,
    IProfilePictureStateProps
> = state => ({
    picturesStatus: state.picturesStatus
});

const mapDispatchToProps: MapDispatchToProps<
    IMeControlAppState,
    IProfilePictureDispatchProps
> = dispatch => ({
    loadPicture(pictureUrl: string, picStatus: IPictureStatus): void {
        dispatch(setPictureUrlAsync(pictureUrl, picStatus));
    },
    fetchMSGraphPicture(currentAccount: IAccountState): void {
        dispatch(getPictureFromGraph(currentAccount));
    }
});

export const ProfilePicture: ComponentFactory<IProfilePictureProps> = connect(
    mapStateToProps,
    mapDispatchToProps
)(ProfilePictureImpl) as any;
