import { h, FunctionalComponent } from 'preact';
import { IAccountState, PageAction } from '@mecontrol/common';
import { getString, createId } from '@mecontrol/web-inline';
import { LinkButton } from './LinkButton';
import { ProfilePicture } from './ProfilePicture';
import { format, isAadWithEditPicUrl, raiseSyntheticEvent } from '../utilities';

export interface IProfilePictureOverlayProps {
    id: string;
    currentAccount: IAccountState;
    glyphClassOverride?: string;
    ariaLabel?: string;
    title?: string;
}

export const ProfilePictureOverlay: FunctionalComponent<IProfilePictureOverlayProps> = props => {
    // Assumption is made that a verification that the profile.editProfilePictureUrl was specified before loading this component.
    // In other cases,
    //   when account type is AAD and editPictureUrl field is present, use the value of editPictureUrl
    //   use the default edit picture url
    let uri = props.currentAccount && props.currentAccount.profile
        ? (props.currentAccount.profile.editProfilePictureUrl)
        : (props.currentAccount && isAadWithEditPicUrl(props.currentAccount))
            ? props.currentAccount.editPictureUrl
            : format('https://account.microsoft.com/profile/edit-picture?username={0}&ref=MeControl', props.currentAccount ? encodeURIComponent(props.currentAccount.memberName) : '');

    return (
        <LinkButton
            id={props.id}
            contentId={PageAction.editPicture}
            cssClass={'mectrl_profilepic_button'}
            getUrl={() => uri}
            ariaLabel={getString('updatephotoa11y')}
            title={getString('updatephotoa11y')}
            onClick={mouseEvent =>
                raiseSyntheticEvent(
                    'photochangeclick',
                    { uri, account: props.currentAccount },
                    mouseEvent
                )
            }
            openInNewTab={true}
        >
            <ProfilePicture
                {...props}
                id={createId(props.id, 'profile_picture')}
            />
        </LinkButton>
    );
};
