import { h, FunctionalComponent, ComponentFactory } from 'preact';
import { bindActionCreators } from 'redux';
import cc from 'classcat';
import { ICommand, ITheme, SignInExperienceType } from '@mecontrol/public-api';
import {
    IMeControlAppState, IMeControlAccounts, IMeControlDisplayMode, IAccountState,
    IAuthProvider, IAuthActions,
    PageName, PageAction, ContentSource,
    AnchorPosition, AnchorAlignment, ToggleAction
} from '@mecontrol/common';
import {
    Promise, SyntheticEvent,
    logTelemetryEvent, createId,
    getOptions, getString, getTimeSinceStart
} from '@mecontrol/web-inline';
import { MapStateToProps, MapDispatchToProps, connect } from '../core/connect';
import { RenderAsync } from '../core/RenderAsync';
import { Header } from './Header';
import { LinkButton } from './LinkButton';
import { FocusVisible } from './FocusVisible';
import { Breakpoint } from './Breakpoint';
import { ScreenReaderText } from './ScreenReaderText';
import { authActions } from '../actions/authActions';
import {
    getCurrentAccount, getOtherAccounts, toIAccount,
    getThemeClass, getHeaderLabel, loadCore, isIE, setUrgent
} from '../utilities';
import { IDropdownProps, IBodyProps } from '@mecontrol/web-core';

export interface RootOwnProps {
    authProvider: IAuthProvider;

    // TODO: add a prop for the SyntheticEventTarget to pass down so that we can use it later in Root for expand/collapse events
}

export interface RootStateProps {
    accounts: IMeControlAccounts;
    commands?: ICommand[];
    displayMode: IMeControlDisplayMode;
    theme: ITheme;
    headerTheme: ITheme;
}

export interface RootDispatchProps extends IAuthActions { }

export type RootProps = RootStateProps & RootDispatchProps & RootOwnProps;

interface MeCoreComponents {
    Body: FunctionalComponent<IBodyProps>;
    Dropdown: FunctionalComponent<IDropdownProps>;
}

const rootClass = 'mectrl_root';
const ieClass = 'mectrl_ie';

const RootImpl: FunctionalComponent<RootProps> = props => {
    // bool value to know if the supported types is flagged as BOTH items independantly (not converged).
    const isBothTypes = ((props.authProvider.supportedSignInAccountTypes & SignInExperienceType.Msa) == SignInExperienceType.Msa
        && (props.authProvider.supportedSignInAccountTypes & SignInExperienceType.Aad) == SignInExperienceType.Aad);
    const currentAccount = getCurrentAccount(props.accounts);
    const cssClass = cc([rootClass, getThemeClass(props.theme), getThemeClass(props.headerTheme, true), { [ieClass]: isIE(navigator.userAgent) }]);
    const ariaLabel = getHeaderLabel(currentAccount);

    const auth = props.authProvider;
    const renderSignIn = renderSignInLinkButton(auth, props.signIn);

    const defaultExpanded = getOptions()?.primaryUXConfig?.defaultExpanded;
    if (defaultExpanded === true) {
        // Immediately download meCore if we are going to immediately expand the
        // dropdown
        setUrgent();
    }

    /* Conditionaly choose to show the dropdown body based on the following critiera:
     * 1) Does the IDP support both types of accounts?  OR
     * 2) Is there a current (active) account?  OR
     * 3) Is there any remembered account types that the auth provider supports 'signInTo' for?
     * */
    let rootContentsRender: (exceeded: boolean) => JSX.Element;
    if (isBothTypes || currentAccount !== undefined || doesSupportSigningInToRemembered(props.authProvider, getOtherAccounts(props.accounts))) {
        rootContentsRender = exceeded => {
            const renderHeader = () => (
                <div>
                    <ScreenReaderText>{ariaLabel}</ScreenReaderText>
                    <Header
                        account={currentAccount}
                        text={getString('signin')}
                        hideText={!exceeded}
                    />
                </div>
            );

            return <RenderAsync
                loader={loadCore as () => Promise<MeCoreComponents>}
                render={meCore =>
                    <meCore.Dropdown
                        id='main'
                        renderTrigger={renderHeader}
                        renderBody={() => <meCore.Body {...props} />}
                        onToggle={handleDropdownStateChange}
                        cssClass='mectrl_mainDropdown'
                        ariaLabel={ariaLabel}
                        tooltip={true}
                        position={AnchorPosition.Bottom}
                        alignment={AnchorAlignment.End}
                        defaultExpanded={defaultExpanded}
                    />
                }
                renderLoading={renderHeader}
                renderError={error => {
                    if (currentAccount) {
                        return renderSignOutLinkButton(auth, currentAccount, props.signOutFromApp)(exceeded);
                    }
                    else {
                        return renderSignIn(exceeded);
                    }
                }}
            />;
        };
    }
    else {
        rootContentsRender = renderSignIn;
    }

    return (
        <FocusVisible cssClass={cssClass}>
            <Breakpoint
                width={props.displayMode.breakpoint}
                mode={props.displayMode.current}
                render={rootContentsRender}
            />
        </FocusVisible>
    );
};
RootImpl.displayName = 'Root';

/**
 * Given the auth provider and list of remembered accounts, this method will loop through each account and determine if
 * at least ONE of the accounts is supported for a 'signInTo' action.
 * @param authProvider The current auth provider for the IDP.
 * @param rememberedAccounts The list of remembered accounts to check.
 */
function doesSupportSigningInToRemembered(authProvider: IAuthProvider, rememberedAccounts: IAccountState[]): boolean {
    if (rememberedAccounts.length > 0) {
        return rememberedAccounts.reduce((isSupported, account) => {
            return isSupported || authProvider.isOperationSupported('signInTo', account.type);
        }, false as boolean);
    }

    return false;
}

/**
 * Function to respond to changes in the dropdown's collapsed state. When called
 * with the current account type (if any) returns the actual handler.
 * Mainly fires telemetry and notifies partners about the changes.
 * @param accountType Type of the current account (if any)
 * @param expanded New state for the dropdown
 * @param action Type of action that triggered the change
 */
const handleDropdownStateChange = (expanded: boolean, action?: ToggleAction) => {
    // First, notify any listeners - in particular, UHF can (will?) use this to handle
    // cookie compliance - they listen for this to know that the user has interacted
    // with the MeControl which implies acceptance of cookie policy

    // TODO: use the passed in (once we pass it down in props) SyntheticEventTarget rather
    // than pulling options out of thin air...
    let meControlConfig = getOptions();
    if (meControlConfig) {
        let dispatchedEvent = new SyntheticEvent(expanded ? "controlexpanded" : "controlcollapsed", undefined);
        meControlConfig.syntheticEventTarget.dispatchEvent(dispatchedEvent);
    }

    // Now fire our own telemtry on this:
    // Log click action that closed/opened the dropdown
    // Interactions are handled by whatever component was interacted with
    if (action !== ToggleAction.Interaction) {
        logTelemetryEvent({
            eventType: 'PageAction',
            content: {
                id: action === ToggleAction.Trigger ? PageAction.openCloseControl : PageAction.dismissControl,
                source: ContentSource.Action
            }
        });
    }

    // Log new page view for the dropdown
    logTelemetryEvent({
        eventType: 'ContentUpdate',
        content: {
            id: expanded ? PageName.expanded : PageName.collapsed,
            source: ContentSource.UX
        },
        loadTime: getTimeSinceStart()
    });
};

/**
 * Function to create another function that renders a SignIn LinkButton
 * @param auth Authprovider from the MeControl configuration
 * @param signIn Sign In Auth action
 */
const renderSignInLinkButton = (auth: IAuthProvider, signIn: IAuthActions['signIn']) =>
    (exceeded: boolean) => {
        const nav = auth.navProvider;
        const label = getString('signinlabel');
        return <LinkButton
            id={createId('main', 'trigger')}
            contentId={PageAction.signIn}
            cssClass='mectrl_trigger'
            ariaLabel={label}
            getUrl={() => nav && nav.getSignInUrl({ signInType: auth.supportedSignInAccountTypes })}
            onClick={(event: Event) => signIn({ signInType: auth.supportedSignInAccountTypes }, event)}
            title={exceeded ? undefined : label}
        >
            <ScreenReaderText>{label}</ScreenReaderText>
            <Header
                text={getString('signin')}
                hideText={!exceeded}
            />
        </LinkButton>;
    };

/**
 * Creates a function that itself renders a sign out link.
 * Only used as a fallback when loading meCore fails and the user is signed in
 * @param auth AuthProvider from the MeControl configuration
 * @param account Currently signed in account
 * @param signOut Sign Out From App action
 */
const renderSignOutLinkButton = (auth: IAuthProvider, account: IAccountState, signOut: IAuthActions['signOutFromApp']) =>
    (exceeded: boolean) => {
        const nav = auth.navProvider;
        const currentAccount = toIAccount(account);
        const label = getString('signout');
        return <LinkButton
            id={createId('main', 'trigger')}
            contentId={PageAction.signOut}
            cssClass='mectrl_trigger'
            ariaLabel={label}
            getUrl={() => nav && nav.getSignOutFromAppUrl({ currentAccount })}
            onClick={(event: Event) => signOut({ currentAccount }, event)}
            title={exceeded ? undefined : label}
        >
            <ScreenReaderText>{label}</ScreenReaderText>
            <Header
                text={getString('signout')}
                hideIcon={true}
            />
        </LinkButton>;
    };

export const mapStateToProps: MapStateToProps<IMeControlAppState, RootStateProps> = state => state;

export const mapDispatchToProps: MapDispatchToProps<IMeControlAppState, RootDispatchProps> = dispatch => {
    return {
        ...bindActionCreators(authActions as any, dispatch)
    };
};

export const Root: ComponentFactory<RootOwnProps> = connect(mapStateToProps, mapDispatchToProps)(RootImpl);
