import { h, Component, render, cloneElement, VNode } from 'preact';
import { createId } from '@mecontrol/web-inline';
import { delayUntilAnimationFrame, getClientRect, getVisibleWindowRect, isElement } from '../utilities';
import { toChildArray } from '../core/toChildArray';

export interface ITooltipProps {
    text: string;
    /**
     * Used when the tooltip should not show regardless of triggers.
     */
    hideOverride?: boolean;
    showToolTipOnClick?: boolean;
}

export interface ITooltipState {
    show: boolean;
    isFocusTriggered: boolean;
    immediate?: boolean;
    timeout?: number;
}

let tooltipCount = 0;
const FourSeconds = 4000;

// Standard tooltips are not accessble via keyboard as illustrated by the
// following bugs;
// - Bug 20651588: [Keyboard Navigation - MeControl - Sign in] Tooltips are not accessible
//      through keyboard on 'More links popup button' under sign in menu.
//      https://microsoft.visualstudio.com/DefaultCollection/OSGS/_workitems/edit/20651588
// - Bug 20636326: [Keyboard Navigation - MeControl - Sign in]Tool tips are not accessible
//      through keyboard on 'More links popup button' under sign in menu.
//      https://microsoft.visualstudio.com/DefaultCollection/OSGS/_workitems/edit/20636326
// Thus, this component
export class Tooltip extends Component<ITooltipProps, ITooltipState> {

    private readonly id: string;
    private element: Element | undefined;

    constructor(props: ITooltipProps) {
        super(props);

        this.state = { show: false, isFocusTriggered: false, immediate: false };

        this.id = createId('tooltip', tooltipCount);
        tooltipCount++;
    }

    /**
     * Handle ESC when the tooltip is showing to dismiss the tooltip.
     * This handler will not interrupt the event, so any other listeners
     * will fire still
     * @param ev Keyboard event object
     */
    private dismissHandler(ev: KeyboardEvent): void {
        if (this.state.show) {
            switch (ev.key) {
                case 'Escape':
                case 'Esc':
                    this.changeVisibility(false);
                    break;
            }
        }
    }

    /** Update visibility of the tooltip */
    private changeVisibility(show: boolean, isFocusTriggered: boolean = false, immediate: boolean = false, timeout: number | undefined = undefined): void {
        this.setState({ show, isFocusTriggered, immediate, timeout });
    }

    private showOnClick(){
        const timeoutId = setTimeout(() => {
            this.changeVisibility(false, false);
        }, FourSeconds);
        this.changeVisibility(true, false, true, timeoutId);
    }

    private removeTooltip(){
        try{
            if(this.state.timeout) {
                clearTimeout(this.state.timeout)
            };
        } catch{}
        this.changeVisibility(false, false);
    }
    
    /** Render DOM element for the tooltip */
    private renderTooltip(show: boolean): void {
        if (show) {
            const tooltip =
                <span id={this.id} class='mectrl_tooltip' role='tooltip' >
                    {this.props.text}
                </span>;
            this.element = render(tooltip, document.body, this.element);
        }
        else if (this.element) {
            this.element = render(null, document.body, this.element);
        }

    }

    public componentDidUpdate(oldProps: ITooltipProps, oldState: ITooltipState): void {
        const { show, isFocusTriggered } = this.state;
        const isFocusVisibleSet = document.querySelector('.mectrl_root.mectrl_focus_visible');

        if(this.props.hideOverride){
            this.renderTooltip(false);
        }
        // Only update/render things if changes where made to state or props
        else if (oldState.show !== show ||
            oldState.isFocusTriggered !== isFocusTriggered ||
            oldProps.text !== this.props.text) {

            // If we should show the tooltip AND either this was not triggered by a focus
            // (which means it was triggered by hover) or if it was and focus-visible is set,
            // then we go ahead and both render the tooltip and position it
            if (show && (!isFocusTriggered || isFocusVisibleSet)) {

                // Render the tooltip. Default styling for it will make it invisible
                this.renderTooltip(true);

                // Delay making the tooltip visible for a bit: 700ms+
                setTimeout(() => {
                    delayUntilAnimationFrame(() => {
                        if (isElement(this.element) && isElement(this.base)) {
                            positionTooltip(this.element as HTMLElement, this.base as HTMLElement, isFocusTriggered);
                        }
                    });
                }, this.state.immediate ? 0 : 700);
            }
            // Otherwise, we remove the tooltip regardless
            else {
                this.renderTooltip(false);
            }
        }
    }

    public componentWillUnmount(): void {
        this.renderTooltip(false);
    }

    public render(): any {
        const children = toChildArray(this.props.children)
            .map(child => {
                return cloneElement(child, {
                    onmouseenter: wrapEventHandler(child, 'onMouseEnter', () => this.changeVisibility(true, false)),
                    onmouseleave: wrapEventHandler(child, 'onMouseLeave', () => this.removeTooltip()),
                    onfocus: wrapEventHandler(child, 'onFocus', () => this.changeVisibility(true, true)),
                    onblur: wrapEventHandler(child, 'onBlur', () => this.changeVisibility(false, true)),
                    onkeydown: wrapEventHandler(child, 'onKeyDown', (ev: KeyboardEvent) => this.dismissHandler(ev)),
                    onclick: wrapEventHandler(child, 'onClick',
                        this.props.showToolTipOnClick
                            ? () => this.showOnClick()
                            : () => this.changeVisibility(false, false)
                    )
                });
            });

        if (children.length > 1) {
            return <div>{children}</div>;
        }
        else {
            return children[0] || null;
        }
    }
}

function positionTooltip(target: HTMLElement, anchor: HTMLElement, isFocusTriggered: boolean): void {
    const vOffset = 7; // px - vertical space between anchor and tooltip
    let targetRect = getClientRect(target);
    let anchorRect = getClientRect(anchor);
    let windowRect = getVisibleWindowRect();

    // Select where to vertically position the tooltip relative to the anchor
    // Preferred position is above with some padding
    let newTop = anchorRect.top >= (targetRect.height + vOffset) ?
        /* Above with spacing */ anchorRect.top - vOffset - targetRect.height :
        /* Below with spacing */ anchorRect.bottom + vOffset;

    // Select where to place the tooltip horizontally
    // Base attempt is to align it to center align it with the anchor
    let newLeft = (anchorRect.width / 2) + anchorRect.left - (targetRect.width / 2);

    // If the tooltip would overflow to the right, adjust it to the right edge
    if (newLeft + targetRect.width > windowRect.right) {
        newLeft = windowRect.right - targetRect.width;
    }
    // If the tooltip would overflow to the left, adjust it to the left edge
    // NOTE: if the tooltip where to overflow in both directions, adjusting to the left
    // edge and then setting right to 'auto' helps alleviate the overflow
    if (newLeft < 0) {
        newLeft = 0;
    }

    // Set styles if possible
    if (target && target.style) {
        target.style.left = `${newLeft}px`;
        target.style.top = `${newTop}px`;
        target.style.right = 'auto';
        target.style.bottom = 'auto';
        target.style.visibility = 'visible';
    }
}

function wrapEventHandler(node: VNode, handlerName: string, newHandler: (ev: any) => void): (ev: Event) => void {
    // Keep the original handler if it was set before
    // Also, try the name as it is passed or all lowercase
    const originalHandler = node.attributes[handlerName]
        || node.attributes[handlerName.toLowerCase()];

    return (ev: Event) => {
        if (originalHandler) {
            originalHandler(ev);
        }
        newHandler(ev);
    };
}
