import { h, Component } from 'preact';
import cc from 'classcat';
import { delayUntilAnimationFrame } from '../utilities';
import { Tooltip } from './Tooltip';

export interface ITruncateTagProps {
    tag: string;
    title: string;
    hideTooltip?: boolean;
    [key: string]: any;
}

export interface ITruncateTagState {
    showTooltip: boolean;
}

// Class used to truncate tag contents
const truncateClass = 'mectrl_truncate';

export class TruncateTag extends Component<
    ITruncateTagProps,
    ITruncateTagState
> {
    constructor(props: ITruncateTagProps) {
        super(props);
        this.state = { showTooltip: false };
    }

    public componentDidMount(): void {
        this.evaluateAndSetShowTooltip();
    }

    public componentDidUpdate(): void {
        this.evaluateAndSetShowTooltip();
    }

    private evaluateAndSetShowTooltip(): void {
        if (this.props.hideTooltip === false) {
            !this.state.showTooltip && this.setState({ showTooltip: true });
        } else if (!this.props.hideTooltip) {
            delayUntilAnimationFrame(() => {
                const element = this.base;
                const showTooltip = element
                    ? element.offsetWidth < element.scrollWidth
                    : false;
                showTooltip !== this.state.showTooltip &&
                    this.setState({ showTooltip });
            });
        }
    }

    public render(): JSX.Element {
        let { tag, title, children, hideTooltip, showToolTipOnClick, ...otherAttributes } =
            this.props;
        const showTooltip = !hideTooltip && this.state.showTooltip;

        let attributes: any = otherAttributes || {};
        attributes.class = cc([attributes.class, truncateClass]);

        const tagNode = h(tag, attributes, children || null);

        if (showTooltip) {
            return <Tooltip text={title} showToolTipOnClick={showToolTipOnClick}>{tagNode}</Tooltip>;
        } else {
            return tagNode;
        }
    }
}
