import { h, FunctionalComponent } from 'preact';
import { AccountType } from '@mecontrol/public-api';
import { IAccountState, PageAction } from '@mecontrol/common';
import { getString, createId, ME } from '@mecontrol/web-inline';
import { LinkButton, raiseSyntheticEvent, setQueryParams } from '@mecontrol/web-boot';

export interface IViewAccountLinkProps {
    account: IAccountState;
}

/**
 * Shows a link to account management. MSA's go to AMC, AAD/MSA_FED goes to azure.
 * @param props The body props with a current account
 */
export const ViewAccountLink: FunctionalComponent<IViewAccountLinkProps> = props => {
    const account = props.account;
    const id = createId('viewAccount'), descriptionId = createId(id, 'desc');
    let uri: string, linkText = '', description = '';

    switch (account.type) {
        case AccountType.MSA:
        case AccountType.MSA_FED:
            uri = setQueryParams(ME.Config.msaUrl, {
                ref: 'MeControl',
                username: account.memberName
            });
            linkText = getString('viewaccount');
            description = getString('msaaccountlinka11y');
            break;
        case AccountType.AAD:
            const params: Record<string, string> = { ref: 'MeControl', login_hint: account.memberName };
            if (account.tenantId) {
                params.tid = account.tenantId;
            }

            uri = setQueryParams(ME.Config.aadUrl, params);
            linkText = getString('viewaccountaad');
            description = getString('aadaccountlinka11y');
            break;
    }

    return (
        <div>
            <LinkButton
                id={id}
                contentId={PageAction.viewAccount}
                getUrl={() => uri}
                onClick={mouseEvent => raiseSyntheticEvent('uriClickEvent', { uri }, mouseEvent)}
                cssClass='mectrl_link'
                aria-describedby={descriptionId}
                openInNewTab={true}
            >
                {linkText}
            </LinkButton>
            <span id={descriptionId} style='display: none;'>{description}</span>
        </div>
    );
};
