import { Component } from "preact";
import { Promise } from "@mecontrol/web-inline";

export interface RenderAsyncState<Response> {
    error?: Error;
    response?: Response;
}

export interface RenderAsyncProps<Response> {
    // ownProps: Props;
    loader(): Promise<Response>;
    render(response: Response): JSX.Element | null;
    renderLoading?(): JSX.Element;
    renderError?(error: Error): JSX.Element;
}

// Prior Art:
// https://github.com/jamiebuilds/react-loadable
// https://github.com/ctrlplusb/react-async-component
// To consider
// * how to handle timeouts
// * delay showing loading component to avoid screen flash
export class RenderAsync<Response> extends Component<
    RenderAsyncProps<Response>,
    RenderAsyncState<Response>
> {
    private loader: Promise<Response> | undefined;
    private mounted: boolean = false;

    constructor(props: RenderAsyncProps<Response>) {
        super(props);
        this.state = {};
    }

    private getLoader(): Promise<Response> {
        if (typeof this.loader === "undefined") {
            try {
                this.loader = Promise.resolve(this.props.loader());
            } catch (error) {
                this.loader = Promise.reject(error);
            }
        }

        return this.loader;
    }

    public componentDidMount(): void {
        this.mounted = true;
        this.getLoader()
            .then(response => {
                if (this.mounted) {
                    this.setState({
                        response
                    });
                }
            })
            .catch(error => {
                if (this.mounted) {
                    this.setState({
                        error
                    });
                }
            });
    }

    public componentWillUnmount(): void {
        this.mounted = false;
    }

    public render(
        props: RenderAsyncProps<Response>,
        state: RenderAsyncState<Response>
    ): JSX.Element | null {
        const { render, renderError, renderLoading } = props;
        const { error, response } = state;

        if (error) {
            return renderError ? renderError(error) : null;
        } else if (response) {
            return render(response);
        } else if (renderLoading) {
            return renderLoading();
        } else {
            return null;
        }
    }
}
