import { Component, ComponentConstructor, h, ComponentFactory, RenderableProps } from "preact";
import { Unsubscribe } from "redux";
import { MeControlDispatch } from '@mecontrol/common';
import { ProviderContext } from "./Provider";

// FUTURE: consider passing ownProps into function if useful. Does change the core logic around when
// to invoke mapping functions though (see "only invokes mapStateToProps..." UT).
export type MapStateToProps<State, StateProps> = (state: State) => StateProps;
export type MapDispatchToProps<State, DispatchProps> = (
    dispatch: MeControlDispatch
) => DispatchProps;

/**
 * Omit from T1, the properties of T2.
 * In other words, remove properties of T2 from T1.
 */
export type Omit<T1, T2> = Pick<T1, Exclude<keyof T1, keyof T2>>;
export type OwnProps<FullProps, InjectableProps> = Omit<FullProps, InjectableProps>;

export type ConnectComponent<FullProps, InjectableProps> = ComponentConstructor<OwnProps<FullProps, InjectableProps>>;

export interface ConnectDecorator<InjectableProps> {
    // tslint:disable-next-line:callable-types
    <FullProps>(
        WrappedComponent: ComponentFactory<FullProps>
    ): ConnectComponent<FullProps, InjectableProps>;
}

export function connect<State, StateProps extends object, DispatchProps>(
    mapStateToProps: MapStateToProps<State, StateProps>,
    mapDispatchToProps: MapDispatchToProps<State, DispatchProps>
): ConnectDecorator<StateProps & DispatchProps> {
    type ActualInjectableProps = StateProps & DispatchProps;

    return function <FullProps>(WrappedComponent: ComponentFactory<FullProps>): ConnectComponent<FullProps, ActualInjectableProps> {
        type ActualOwnProps = OwnProps<FullProps, ActualInjectableProps>;

        const wrappedComponentName = WrappedComponent.displayName
            || (WrappedComponent as any).name
            || 'Component';

        // TODO: Pass down ownProps to selectors and memoizing the results, using those
        //       results to determine shouldComponentUpdate. Will need to run selector on
        //       componentWillReceiveProps
        // FUTURE: If a component needs a custom merging strategy, add the mergeProps param
        // FUTURE: Consider hoisting statics from WrappedComponent if necessary
        // FUTURE: Consider exposing WrappedComponent as static property WrappedComponent
        // FUTURE: Consider adding ability to access wrapped instance
        // FUTURE: Consider option to receive/pass store as a prop?

        // HOC = Higher Order Component. Bing it.
        return class ConnectHOC extends Component<ActualOwnProps, any> {
            public static displayName: string = `Connect(${wrappedComponentName})`;

            private unsubscibe: Unsubscribe | undefined;
            private dispatchProps: DispatchProps;
            private stateProps: StateProps;

            constructor(props: any, context: ProviderContext<State>) {
                super(props, context);
                const store = context.store;
                if (!store) {
                    throw new Error("MeControl: ConnectHOC - store not found in context");
                }

                this.stateProps = mapStateToProps(store.getState());
                this.dispatchProps = mapDispatchToProps(store.dispatch);
            }

            public componentDidMount(): void {
                const context = this.context as ProviderContext<State>;
                this.unsubscibe = context.store.subscribe(() => this.update());
            }

            public componentWillUnmount(): void {
                if (this.unsubscibe) {
                    this.unsubscibe();
                }
            }

            public render(ownProps: RenderableProps<ActualOwnProps>): JSX.Element {
                const finalProps = {
                    ...(this.dispatchProps as any),
                    ...(ownProps as any),
                    ...(this.stateProps as any)
                };

                return h(WrappedComponent, finalProps);
            }

            private update(): void {
                const context = this.context as ProviderContext<State>;
                const store = context.store;

                const oldStateProps = this.stateProps;
                const newStateProps = mapStateToProps(store.getState());

                // IE9 - Object.keys
                const oldStatekeys = Object.keys(oldStateProps);
                const newStateKeys = Object.keys(newStateProps);

                // Detect if keys were added or removed
                if (newStateKeys.length !== oldStatekeys.length) {
                    this.stateProps = newStateProps;
                    return this.setState({});
                }

                // Determine if any keys' values were changed
                for (let newKey in newStateProps) {
                    if (newStateProps[newKey] !== oldStateProps[newKey]) {
                        this.stateProps = newStateProps;
                        return this.setState({}); // queue re-render of this component and children
                    }
                }
            }
        };
    };
}
