import { Reducer } from 'redux';
import {
    IPictureStatusDictionary, PictureLoadStatus, AllActions, ActionTypes
} from '@mecontrol/common';
import { GRAPH_PIC_STATUS_KEY } from '../actions/pictureActions';
import { CACHED_DONE_KEY, REMEMBERED_DONE_KEY } from '../actions/cacheActions';

export const picturesStatus: Reducer<IPictureStatusDictionary> = (state = {}, action: AllActions) => {
    switch (action.type) {
        case ActionTypes.REMEMBERED_ACCOUNTS_START:
            return UpdateStatus(state, REMEMBERED_DONE_KEY, PictureLoadStatus.Loading);
        case ActionTypes.REMEMBERED_ACCOUNTS_FAILED:
            return UpdateStatus(state, REMEMBERED_DONE_KEY, PictureLoadStatus.Failed);
        case ActionTypes.REMEMBERED_ACCOUNTS_COMPLETED:
            return UpdateStatus(state, REMEMBERED_DONE_KEY, PictureLoadStatus.Succeeded);
        case ActionTypes.GET_CACHED_COMPLETED:
        case ActionTypes.GET_CACHED_DISABLED:
        case ActionTypes.GET_CACHED_PICTURES_BLANK:
        case ActionTypes.GET_CACHED_PICTURES_SUCCESS:
            return UpdateStatus(state, CACHED_DONE_KEY, PictureLoadStatus.Succeeded);
        // Update state after the account picture has been refreshed
        case ActionTypes.REFRESH_PICTURE:
            return UpdateStatus(state, action.payload.url, PictureLoadStatus.None);

        // Update state after the profile picture has loaded successfully
        case ActionTypes.LOAD_PICTURE_URL_SUCCESS:
            return UpdateStatus(state, action.payload.pictureUrl, PictureLoadStatus.Succeeded);

        // Update the state when the profile picture failed to load
        case ActionTypes.LOAD_PICTURE_URL_FAILED:
            return UpdateStatus(state, action.meta.pictureUrl, PictureLoadStatus.Failed);

        // Update the state when the profile picture started loading
        case ActionTypes.LOAD_PICTURE_URL_START:
            return UpdateStatus(state, action.payload.pictureUrl, PictureLoadStatus.Loading);

        case ActionTypes.GET_GRAPH_PICTURE_START:
            return UpdateStatus(state, GRAPH_PIC_STATUS_KEY, PictureLoadStatus.Loading);
        case ActionTypes.GET_GRAPH_PICTURE_SUCCESS:
            return UpdateStatus(state, GRAPH_PIC_STATUS_KEY, PictureLoadStatus.Succeeded);
        default:
            return state;
    }
};

/**
 * Helper method to update the state based on the specified status.
 * @param state The state being updated
 * @param pictureUrl The URL key that is being updated by cound and status.
 * @param status The status to adjust the pictureStatus to be.
 */
function UpdateStatus(state: IPictureStatusDictionary, pictureUrl: string | undefined, status: PictureLoadStatus): IPictureStatusDictionary
{
    if (!pictureUrl) {
        return state;
    }

    // Set default start for tries count.
    let tries: number = 0;

    // Try and get any previous state value for the picture Url and attain the current tries for that item.
    let picStatus = state[pictureUrl];

    if (picStatus) {
        tries = picStatus.tries;
    }

    if (status === PictureLoadStatus.None) {
        if (picStatus && picStatus.status === PictureLoadStatus.Succeeded) {
            return state;
        }
        tries = 0;
    }

    // Increment if loading started
    if (status === PictureLoadStatus.Loading) {
        tries++;
    }

    // Return updated state and status.
    return {
        ...state,
        [pictureUrl]: {
            status,
            tries
        }
    };
}
