import { Middleware } from 'redux';
import {
    IMeControlAppState, IMeControlAccounts, FluxStandardAction, IClientErrorParams,
    ActionTypes, ErrorSeverity, IPictureStatusDictionary
} from '@mecontrol/common';
import { logTelemetryEvent, hasOwn, toJsonable, scrubUrl } from '@mecontrol/web-inline';
import { filterObject } from '../utilities/telemetry';

/**
 * Error handling middleware that catches any errors in our reducers and logs them
 * using our telemetry pipeline
 * @param store Store instance used in the middleware
 * @param next The middleware-enhanced version of dispatch that this middleware will wrap around.
 * This is what should be called to actually dispatch the passed-in action
 * @param action Redux action to be dispatched
 */
export const errorHandler: Middleware = store => next => action => {
    // Dispatch the action that was received
    // If the action is an error action (except for picture load ones, which are expected)
    // we log that to our telemetry pipeline
    try {
        if (action.error && action.type !== ActionTypes.LOAD_PICTURE_URL_FAILED) {
            logErrors(store.getState(), action);
        }

        return next(action);
    }
    // The catch block will handle firing any telemetry for errors inside of reducers
    catch (err) {
        logErrors(store.getState(), action, err);

        // Only re-throw the error if we are in a dev environment
        if (process.env.NODE_ENV === 'development') {
            //tslint:disable-next-line:no-console
            console.error(err);
        }
    }
};

/**
 * Log telemetry on errors in our Redux app. If an error, is explicitly passed, then its information
 * will be included alongside the current state and action. Otherwise, just the action and state will
 * be included.
 * @param state Current Redux state when the error handler was invoked
 * @param action Action that was being dispatched when the error occurred
 * @param error Possible error that is being logged
 */
function logErrors(state: IMeControlAppState, action: FluxStandardAction, error?: Error): void {
    let { state: fState, action: fAction } = filterStateAndAction(state, action);
    let errorParams: IClientErrorParams;
    let errorDetails: any = {
        state: fState,
        action: fAction
    };

    // When an actual error is being logged, we extract it's details
    // for the ClientError event
    if (error) {
        errorDetails.stackTrace = error.stack;
        errorParams = {
            eventType: 'ClientError',
            name: error.message,
            type: 'ReducerError',
            details: JSON.stringify(errorDetails),
            displayed: false
        };
    }
    // For FAIL Redux actions, we use as message the one from the action itself
    else if (action.error) {
        let actionErr = action.payload as Error;
        errorParams = {
            eventType: 'ClientError',
            name: actionErr.message,
            type: `FailedAction.${action.type}`,
            details: JSON.stringify(errorDetails),
            displayed: true,
            severity: ErrorSeverity.Warning
        };
    }
    // This case should not happen, but we handle it to catch strange errors
    else {
        errorParams = {
            eventType: "ClientError",
            name: 'No error passed to error handler middleware',
            type: 'UnkownErrorHandlerCall',
            details: JSON.stringify(errorDetails),
            displayed: false,
        };
    }

    logTelemetryEvent(errorParams);
}

// Fields to NOT filter out of account state for telemetry
// These are basically our own redux state variables and the type and sign-in status of the account
const accountFilters = ['*', '!type', '!authenticatedState', '!accountItemStatus'];

// Filed to NOT filter out of actions for telemetry
// We won't filter accountId because the filtering for that is different (see filterStateAndAction())
const actionFilters = ['*', '!type', '!error', '!accountId'];

interface StateAndAction {
    state: IMeControlAppState;
    action: FluxStandardAction;
}

/**
 * Filters out the current state and fired action for the purposes of telemetry logging. This means that all
 * PII will be filtered out from the accounts part of state (while the rest of it, which is internal and not
 * sensitive remains) as well as the action. AccountIds will be replaced with a number and be consistent with
 * their previous values (meaning that currentId will match an id in byId and if the action had an account ID
 * from a known account, that will match too)
 * @param currentState State of the app when an error occurred
 * @param action Action that was being fired when an error occurred
 */
function filterStateAndAction(currentState: IMeControlAppState, action: FluxStandardAction): StateAndAction {
    const accounts = currentState.accounts;
    const commands = currentState.commands;
    const picturesStatus = currentState.picturesStatus;

    // Used to replace account IDs, which are sensitive, with numerical ones
    const accountMap = createTranslationMap();

    // Used to replace picture URLs, which may contain sensitive data
    const pictureMap = createTranslationMap();

    // Object that will be populated with filtered data from the state
    let filteredAccounts: IMeControlAccounts = {
        byId: {},
        allIds: [],
        currentId: ''
    };

    // Filter byId portion of state
    //tslint:disable-next-line:forin
    for (let accountId in accounts.byId) {
        if (hasOwn(accounts.byId, accountId)) {
            // Log the Id for conversion to a numerical one
            // Then filter the account
            // Then log the pictureUrl for conversion to number (if there is one)
            let account = accounts.byId[accountId];
            let newId = accountMap.getValue(accountId);
            filteredAccounts.byId[newId] = filterObject(account, accountFilters) as any;
            if (account.pictureUrl) {
                filteredAccounts.byId[newId].pictureUrl = pictureMap.getValue(account.pictureUrl);
            }
        }
    }

    // Filter allIds
    // We do this separately in case there was a lack of consistency between byId and allIds
    // (which could be a part of the error)
    filteredAccounts.allIds = accounts.allIds.map(accountMap.getValue);

    // Finally, filter our currentId
    filteredAccounts.currentId = accounts.currentId && accountMap.getValue(accounts.currentId);

    // Filter commands URLs (just in case)
    let filteredCommands = commands && commands.map(com => ({
        ...com,
        url: typeof com.url === 'string' ? scrubUrl(com.url) : com.url
    }));

    // Filter picture URLs (just in case too)
    let filteredPicturesStatus: IPictureStatusDictionary = {};
    //tslint:disable-next-line:forin
    for (let pictureUrl in picturesStatus) {
        if (hasOwn(picturesStatus, pictureUrl)) {
            let newPictureUrl = pictureMap.getValue(pictureUrl);
            filteredPicturesStatus[newPictureUrl] = picturesStatus[pictureUrl];
        }
    }

    // Filter the action
    // Error actions will restore the payload, as it contains an error
    // For accountId, we will use the idMap to replace it with a numerical id
    let filteredAction = filterObject(action, actionFilters) as FluxStandardAction;
    if (action.error) {
        // Restore payload, as it contains error information
        // It requires special JSON handling though
        filteredAction.payload = toJsonable(action.payload);
        if (action.meta && action.meta.accountId) {
            filteredAction.meta.accountId = accountMap.getValue(action.meta.accountId);
        }
    }
    else {
        if (action.payload && action.payload.accountId) {
            filteredAction.payload.accountId = accountMap.getValue(action.payload.accountId);
        }
    }

    return {
        state: {
            ...currentState,
            accounts: filteredAccounts,
            commands: filteredCommands,
            picturesStatus: filteredPicturesStatus
        },
        action: filteredAction
    };
}

/**
 * Create a map to store values and map them to numbers as a translation
 * to scrub the original values away
 */
const createTranslationMap = () => {
    let internalMap: Record<string, number> = {};
    let currentValue = 0; // Value to add to the map

    return {
        getValue: (value: string) => {
            // Any values that are not found in the map, are added to it
            let mappedValue = internalMap[value];
            if (mappedValue === undefined) {
                internalMap[value] = mappedValue = currentValue++;
            }
            return mappedValue.toString();
        }
    };
};
