import { IAccount, AuthenticatedState, AccountType, IAADAccount, IFederatedMSAAccount, IMSAAccount, IAccountBase } from '@mecontrol/public-api';
import { IMeControlAccounts, IAccountState, IMeControlAppState } from '@mecontrol/common';
import { getString, ME } from '@mecontrol/web-inline';
import { isNonEmptyString } from './types';
import { format } from './string';
import { pick } from './pick';
import { find } from './find';

export interface AnyAccount extends IAccountBase {
    [key: string]: any;
}

//============================
// General Account Util
//============================

/**
 * Return true if the given account is AAD-like, meaning it has the shape of an
 * AAD account, namely AccountType.AAD and AccountType.MSA_FED
 * @param account The account to check
 */
export function isAadLike(account: IAccount): account is (IAADAccount | IFederatedMSAAccount) {
    return account.type === AccountType.AAD || account.type === AccountType.MSA_FED;
}

/**
 * Return true if the given account is functionally MSA-like, meaning it has the shape of an
 * MSA account, namely AccountType.MSA and AccountType.MSA_FED
 * @param account The account to check
 */
export function isFunctionallyMsaLike(account: IAccount): account is (IMSAAccount | IFederatedMSAAccount) {
    return account.type === AccountType.MSA || account.type === AccountType.MSA_FED;
}

/**
 * Return true if the given account type is AAD and has a valid editPictureUrl field
 * @param account The account to check
 */
export function isAadWithEditPicUrl(account: IAccount): boolean {
    return account.type === AccountType.AAD && !!account.editPictureUrl;
}

/**
 * Returns true if the given account has a profile AND that profile contains a valid edit picture URL.
 * If no profile is specified,
 *    then true is returned
 *      if the account is MSA like.
 *      or if the account type is AAD and has a valid editPictureUrl field
 * @param account The account to check.
 */
export function isFunctionallyMsaOrWithEditPicUrl(account: IAccount): boolean {
    if (account.profile) {
        /* If profile present, ONLY return true if the edit picture URL is present, otherwise false. Do not fall
         * back to MSA-like since a profile was specified. */
        return isNonEmptyString(account.profile.editProfilePictureUrl);
    }

    return isFunctionallyMsaLike(account) || isAadWithEditPicUrl(account);
}

/**
 * Given an IAccount object, this helper method will concat the type and membername
 * into a string used for a unique ID.
 * @param account The account to get a unique ID for.
 * @returns The account's computed ID
 */
export function getAccountId(account: IAccount): string {
    if (account.type === AccountType.MSA && account.cid) {
        const normalizedCid = account.cid.toLowerCase().replace(/^0+/g, '');
        return `${account.type}:${normalizedCid}`;
    }

    return `${account.type}:${account.memberName && account.memberName.toLowerCase()}`;
}

/**
 * Check whether an account is signed in or not
 * @param account Account to check the status of
 */
export function isSignedIn(account: IAccount): boolean {
    return account.authenticatedState === AuthenticatedState.SignedIn ||
        account.authenticatedState === AuthenticatedState.SignedInIDPOnly;
}

/**
 * Get the friendly name to display for a remembered account:
 *  - MSA : compute display name
 *  - AAD : company name
 * If neither is found, we default to the member name of the account
 * @param account Account for which to retrive the name
 */
export function getAccountListDisplayName(account: IAccount): string {
    if (account.profile && account.profile.profileName) {
        return account.profile.profileName;
    }

    switch (account.type) {
        case AccountType.MSA:
        case AccountType.MSA_FED:
            return getDisplayName(account) || account.memberName;

        case AccountType.AAD:
            return account.orgName || getString('aadaccountorgnamedefault');
    }
}

/**
 * Get the friendly name for accessibility purposes only:
 * This friendly name would be assigned to aria-labels only
 *  - MSA : compute display name
 *  - AAD : company name
 * We will concatenate both the display or company name with member name
 * @param account Account for which to retrive the name
 */
export function getVerbalFriendlyName(account: IAccount): string {
    let name: string | undefined;
    switch (account.type) {
        case AccountType.MSA:
        case AccountType.MSA_FED:
            name = getDisplayName(account);
            break;

        case AccountType.AAD:
            name = account.orgName;
            break;
    }
    return name ? name + ' ' + account.memberName : account.memberName;
}

/**
 * Compute the name to display for a given account
 * @param account Account from which to compute the display name
 */
export function getDisplayName(account: IAccount): string | undefined {
    const lf = (first: string | undefined, last: string | undefined) => {
        return ((ME.Loc.lf ? [last, first] : [first, last])
            .map(str => str && str.trim())
            .filter(str => !!str)
            .join(' ')) || undefined;
    };

    let displayName: string | undefined;
    switch (account.type) {
        case AccountType.MSA:
            displayName = lf(account.firstName, account.lastName) || account.displayName;
            break;

        case AccountType.AAD:
        case AccountType.MSA_FED:
            displayName = account.displayName || lf(account.firstName, account.lastName);
            break;
    }
    return displayName && displayName.trim();
}

/**
 * Get the header display name for the current account from the account state
 * @param account Account state from which to extract the active account (if available)
 * @returns The header display name from the state or an empty string if there is none.
 */
export function getHeaderDisplayName(account: IAccountState | undefined): string | undefined {
    if (account) {
        if (account.profile && account.profile.profileName) {
            return account.profile.profileName;
        }
        else {
            switch (account.type) {
                case AccountType.AAD:
                case AccountType.MSA_FED:
                    return account.displayName || account.firstName;

                case AccountType.MSA:
                    return account.firstName || account.displayName;
            }
        }
    }
    return undefined;
}

/**
 * Get the header a11y label
 * @param account Account state for the current account (if any)
 * @returns The header label to use for screen readers
 */
export function getHeaderLabel(account: IAccountState | undefined): string {
    if (account) {
        return format(
            getString('headera11yformat1'),
            getHeaderDisplayName(account) || account.memberName
        );
    }

    return getString('signinlabel');
}

/**
 * Get the full display name for the current account from the account state
 * @param account Account state from which to extract the active account (if available)
 * @returns The full display name from the state or an empty string if there is none.
 */
export function getFullDisplayName(account: IAccountState): string {
    if (account.profile && account.profile.profileName) {
        return account.profile.profileName;
    }

    return getDisplayName(account) || '';
}

/**
 * Determine whether the given list contains the given account
 * @param list A list of accounts
 * @param account The account to search for
 */
export function findAccount(list: IAccount[], account: IAccount): IAccount | undefined {
    const searchId = getAccountId(account);
    return list.filter(account2 => getAccountId(account2) === searchId)[0];
}

//============================
// Current Account
//============================

/**
 * Get the current account from the account state
 * @param accounts Account state from which to extract the active account (if available)
 * @returns The current account from the state or undefined if there is none.
 */
export function getCurrentAccount(accounts: IMeControlAccounts): IAccountState | undefined {
    if (accounts.currentId !== '') {
        return accounts.byId[accounts.currentId];
    }

    return undefined;
}

/**
 * Given the current account object, determine the appropriate URL to return for the users account
 * picture URL. This method will return an empty string if no image url found.
 * @param currentAccount The current account used to display the user's info.
 */
export function getPictureUrl(currentAccount: IAccount | undefined): string {
    if (currentAccount) {
        return currentAccount.profile ?
            currentAccount.profile.profilePictureUrl :
            currentAccount.pictureUrl as string;
    }

    return '';
}

/**
 * Get the current accunt's type. Returns undefined if there is no current account
 * @param state MeControl state to extract the current account from
 */
export function getCurrentAccountType(state: IMeControlAppState): AccountType | undefined {
    let currentAccount = getCurrentAccount(state.accounts);
    return currentAccount && currentAccount.type;
}

//============================
// Non-Active Accounts
//============================

/**
 * Given the account state, get all of the accounts that are NOT the active one (if there is one)
 * @param accounts Account state
 * @returns Array of accounts that are NOT the active one (if any)
 */
export function getOtherAccounts(accounts: IMeControlAccounts): IAccountState[] {
    const types = ['msaFed', 'msa'];
    const activeAccId = accounts.currentId;
    const activeAccMemberName = accounts.byId[activeAccId]?.memberName;
    let otherAccounts: IAccountState[] = [];

    accounts.allIds.forEach(id => {
        let account = accounts.byId[id];
        if (id === activeAccId) {
            // Don't include active account
            return;
        }
        else if (activeAccMemberName != undefined && account.memberName.toLowerCase() === activeAccMemberName.toLowerCase()
            && types.indexOf(account.type) > -1
            && types.indexOf(accounts.byId[activeAccId].type) > -1) {
            // Don't include duplicate msa/msaFed accounts if they have the same
            // memberName
            return;
        }
        else if (find(otherAccounts, other => {
            return account.memberName === other.memberName
                && types.indexOf(account.type) > -1
                && types.indexOf(other.type) > -1;
        }) !== undefined) {
            return;
        } else {
            otherAccounts.push(account);
        }
    });

    return otherAccounts;
}

export function toIAccount(account: AnyAccount): IAccount {
    return pick(account, [
        // General Props
        "type",
        "memberName",
        "authenticatedState",
        "firstName",
        "lastName",
        "displayName",
        "pictureUrl",
        "profile",

        // MSA Props
        "cid",

        // AAD props
        "bearerToken",
        "orgName",
        "orgLogoUrl",
        "sessionId",
        "roleName",
    ]) as IAccount;
}
