
/** Generic event handler */
export type EventHandler<T> = (event: T) => void;

/** Event handler that delays the actual logic for handling events to avoid repeated calls */
export interface DebounceEventHandler<T> {
    /** Actual event handler function */
    (ev: T): EventHandler<T>;
    /** Function to cancel the current queued call for debounced events (if any) */
    cancel: () => void;
}

/**
 * Create an event handler that delays actually handling events in order to avoid excessive computations.
 * When actually handling an event, only the latest instance of said event is passed to the handler.
 * @param eventHandler The actual function that handles the event
 * @param delayMs How long to delay handling events (and accumulating them) for
 */
export function debounce<T>(this: any, eventHandler: EventHandler<T>, delayMs: number = 100): DebounceEventHandler<T> {
    // Variables to keep track of events we are debouncing
    let currentTimeout: number | undefined;
    let lastEvent: T | undefined;

    // Reset state for next debounced event
    const clearState = () => { currentTimeout = lastEvent = undefined; };

    let debouncedEventHandler: any = (event: T) => {
        lastEvent = event;

        // Set up the next call of the actual event handling
        if (!currentTimeout) {
            currentTimeout = setTimeout(() => {
                // Trigger event
                eventHandler.call(this, lastEvent!);
                clearState();
            }, delayMs);
        }
    };

    // Handle canceling the next timeout
    debouncedEventHandler.cancel = () => {
        clearTimeout(currentTimeout);
        clearState();
    };

    return debouncedEventHandler;
}
