import { ME } from '@mecontrol/web-inline';

/**
 * Check if a given element is visible.
 * Works similar to jQuery visible():
 * https://github.com/jquery/jquery/blob/899c56f6ada26821e8af12d9f35fa039100e838e/src/css/hiddenVisibleSelectors.js
 * @param element Element to check
 */
export function isVisible(element: HTMLElement): boolean {
    const rects = element.getClientRects();
    const style = getComputedStyle(element);
    return (
        element.offsetWidth > 0 ||
        element.offsetHeight > 0 ||
        rects.length > 0
    ) &&
        style.visibility !== 'hidden' &&
        style.display !== 'none' &&
        // Only consider opacity if it is defined, else we assume default value of 1
        (style.opacity ? parseFloat(style.opacity) : 1) > 0;
}

/**
 * Find focusable elements that are children of the passed in container element.
 * A more thorough and complete algorithm for this can be found in jquery-ui:
 * https://github.com/jquery/jquery-ui/blob/master/ui/focusable.js
 * Additional documentation here:
 * https://api.jqueryui.com/focusable-selector/
 * @param container Element inside of which the search will be performed
 * @param includeCodeFocus Include elements that can only be focused programatically
 * (i.e. tabindex="-1")
 */
export function getFocusableElements(container: HTMLElement, includeCodeFocus?: boolean): HTMLElement[] {
    // Tag selectors for elements that MAY be focusable
    // AREA is not currently handled
    let focusableTags = [
        'input',
        'select',
        'textarea',
        'button',
        'object',
        'a[href]', // Anchor tags need an href to be focusable
        '[tabindex="0"]' // Anything with tab-index = 0
    ];

    if (includeCodeFocus) {
        focusableTags.push('[tabindex="-1"]');
    }

    // Selector for non-disabled elements
    const enabled = ':not([disabled])';

    // Get all possibly focusable tags from the container
    // Before doing so, we must also ensure they are not disabled
    let possibleFocusTags = container.querySelectorAll(
        focusableTags.map(tag => tag + enabled).join(',')
    );

    // Convert NodeList to Array and ensure it only contains visible elements
    let focusableElements: HTMLElement[] = [];
    for (let i = 0; i < possibleFocusTags.length; i++) {
        let focusableEl = possibleFocusTags[i] as HTMLElement;
        if (isVisible(focusableEl)) {
            focusableElements.push(focusableEl);
        }
    }

    return focusableElements;
}

/**
 * Verify whether a given element should close a dropdown when interacted with
 * @param element DOM Element we are checking
 */
export function shouldCloseDropdown(element: Element): boolean {
    return element.hasAttribute &&
        !element.hasAttribute('data-noClose') &&
        !element.hasAttribute('disabled') &&
        (element.nodeName === 'BUTTON' || element.nodeName === 'A');
}

/** Determine whether the page is in Right To Left mode or not */
export function isRtl(): boolean {
    return ME.Loc.rtl || document.dir === 'rtl';
}

/**
 * Function that runs an unspecified process with the supplied Element.
 * @param element DOM Element on which to run the process
 * @returns Either a boolean value or undefined. If a boolean is returned, and that
 * value is true, then this means the process does not need to keep running for
 * more Elements
 */
export type ProcessElement = (element: Element) => boolean | undefined;

/**
 * Traverse the DOM tree upwards calling a function in each node.
 * @param node Element from which to start DOM tree traversal.
 * @param processNode Function that is called at every node of the tree with the element at that node.
 * If the function returns true, the traversal stops.
 * @returns The node for which the process returned true or null if none did.
 */
export function traverseDomTree(node: Element | undefined, processNode: ProcessElement): Element | undefined {
    // Return the node if it is null or if calling the function on it returned true
    if (!node || processNode(node) === true) {
        return node;
    }

    return traverseDomTree(node.parentNode as Element || undefined, processNode);
}
