import { AnchorAlignment, AnchorPosition } from '@mecontrol/common';
import { Promise } from '@mecontrol/web-inline';
import {
    delayUntilAnimationFrame,
    getClientRect, getVisibleWindowRect, getDocumentRect, isElement
} from '@mecontrol/web-boot';
import { isRtl, traverseDomTree } from './dom';

/** Defines a width and height that an HTML element occupies */
interface ClientArea {
    width: number;
    height: number;
}

/** The default anchor buffer (in pixels) to apply to the outer most size of the drop down. Negative shrinks, positive expands. */
const anchoringBuffer: number = -10;

/**
 * Predefined cycle of positions that defines the order in which different ones
 * are tried when anchoring elements optimally
 */
const positionOrder = [
    AnchorPosition.Left,
    AnchorPosition.Top,
    AnchorPosition.Right,
    AnchorPosition.Bottom
];

/**
 * Place a given Element such that it is anchored to a second one, following the provided
 * position and alignment and ensuring, as much as possible, that it does not overflow its
 * container (which, when not provided, defaults to the visible window). See algorithm
 * description for more details.
 * @param target Element being positioned
 * @param anchor Element around which the target shall be positioned
 * @param position Initial position to try placing the target to
 * @param alignment Alignment of the target with relation to the anchor we will try
 * @param container Optional element against which the positioning will be checked for overflows
 * @param allowOverlap Optional value that will allow positioning of the target to overlap the
 * anchor (in the case where there is not enough space for an optimal position)
 */
export function anchorElementTo(
    target: HTMLElement,
    anchor: HTMLElement,
    position: AnchorPosition,
    alignment: AnchorAlignment,
    container?: HTMLElement,
    allowOverlap: boolean = false
): Promise<void> {

    return new Promise(resolve => {

        // Wait before performing all of this given that we will triger a a style recalculation
        delayUntilAnimationFrame(() => {

            // Determine what should be used as our bounding box:
            //  - The passed in container
            //  - The visible window (for fixed elements)
            //  - The full page (for all other cases)
            // Further down, we will indiscriminately refer to this bounding box as the 'container'
            let boundingBox: ClientRect;
            if (container) {
                boundingBox = getClientRect(container, anchoringBuffer);
            }
            else if (isFixedElement(target)) {
                boundingBox = getVisibleWindowRect();
            }
            else {
                boundingBox = getDocumentRect();
            }

            // Now get the rects for the target and anchor
            let targetRect = getFullClientRect(target);
            let anchorRect = getClientRect(anchor);

            if (targetRect.height !== 0 && targetRect.width !== 0) {

                // Handle RTL by mirroring Position Left <-> Right
                // or for Position Top/Bottom, Alignment Start <-> End
                if (isRtl()) {
                    switch (position) {
                        case AnchorPosition.Left:
                            position = AnchorPosition.Right;
                            break;
                        case AnchorPosition.Right:
                            position = AnchorPosition.Left;
                            break;
                        default:
                            alignment = alignment === AnchorAlignment.Start ?
                                AnchorAlignment.End :
                                AnchorAlignment.Start;
                    }
                }

                // Find and apply optimal position for target
                // Height will only be adjusted if necessary
                let optimalPosition = findOptimalPosition(
                    targetRect,
                    anchorRect,
                    boundingBox, // henceforth known as 'container'
                    position,
                    alignment,
                    allowOverlap
                );
                applyOptimalPosition(
                    target,
                    optimalPosition,
                    targetRect.height != optimalPosition.height
                );
            }

            resolve();
        });
    });
}

/**
 * Given a target and anchor around which to position it, find the best possible location for
 * the target such that is visible in the passed-in container. This method will assign a higher
 * value to making the target visible over respecting the preferred position and alignment values
 * provided.
 * @param target Element being positioned
 * @param anchor Element around which the target shall be positioned
 * @param container Area within which the target must be constrained
 * @param preferredPosition Position that will be given priority when placing the target
 * @param preferredAlignment Alignment that will be given prioritu when placing the target
 * @param allowOverlap Whether to allow the target to wind up covering the anchor -partially or
 * totally- in the case where there is no optimal position that fits the whole target
 */
function findOptimalPosition(
    target: ClientRect,
    anchor: ClientRect,
    container: ClientRect,
    preferredPosition: AnchorPosition,
    preferredAlignment: AnchorAlignment,
    allowOverlap: boolean
): ClientRect {

    // Set up initial values to loop through all possible positions
    let currentPosition = preferredPosition;
    let positionIndex = positionOrder.indexOf(preferredPosition);

    // These values will help keep track of the best candidate in
    // case there is no position that perfectly fits the target
    let bestVerticalFit = -Infinity;
    let bestPosition: AnchorPosition | undefined;

    // Now we actually test each of the positions
    do {
        // For each one, we compute whether the target would fit in the available
        // space in the container
        let verticalFit = computeVerticalFit(target, anchor, container, currentPosition);

        // A positive value means the target will fit without issues, so we can
        // just return that position
        if (verticalFit >= 0) {
            bestPosition = currentPosition;
            break;
        }

        // If we hit these lines, we have not yet found a full fit for the target,
        // so we will keep track of the best candidate position: the one with the
        // highets possible fit value
        if (verticalFit > bestVerticalFit) {
            bestVerticalFit = verticalFit;
            bestPosition = currentPosition;
        }

        positionIndex = (positionIndex + 1) % positionOrder.length;
        currentPosition = positionOrder[positionIndex];
    }
    while (currentPosition !== preferredPosition);

    // When we reach this point, we have a few possibilities:
    //  - We found a good fit, and 'bestPosition' has the value for that
    //  - We found no perfect fits, but we have good candidate that maximizes
    // visible area in 'bestPosition'
    //  - We found nothing at all, and will default to our 'preferredPosition'
    // since nothing really works
    // Considering this, we will simply compute the placement of the target
    // using this knowledge and return that
    return computePlacement(
        target,
        anchor,
        container,
        bestPosition || preferredPosition,
        preferredAlignment,
        allowOverlap
    );
}

/**
 * Compute the ClientRect representing the final global coordinates for a given target, so that
 * it is positioned around an anchor in the passed-in position and alignment values. The final
 * location may tweak height of the target or, based on overflows, not exactly match the position
 * and alignment that were passed in.
 * @param target Element being positioned
 * @param anchor Element around which the target shall be positioned
 * @param container Area within which the target must be constrained
 * @param position Position to place the target in with relation to the anchor
 * @param alignment Alignment of the target with relation to the anchor we will try
 * @param allowOverlap Whether to allow the target to wind up covering the anchor -partially or
 * totally- in the case where there is no optimal position that fits the whole target
 */
function computePlacement(
    target: ClientRect,
    anchor: ClientRect,
    container: ClientRect,
    position: AnchorPosition,
    alignment: AnchorAlignment,
    allowOverlap: boolean
): ClientRect {

    let targetLeft: number = 0;
    let targetTop: number = 0;
    let targetBottom: number = 0;

    // Compute where the top and left of the positioned target should
    // be according to the Position and Alignment values
    switch (position) {
        case AnchorPosition.Left:
            targetLeft = anchor.left - target.width;
            targetTop = alignment === AnchorAlignment.Start ?
                anchor.top :
                anchor.bottom - target.height;
            break;

        case AnchorPosition.Top:
            targetLeft = alignment === AnchorAlignment.Start ?
                anchor.left :
                anchor.right - target.width;
            targetTop = anchor.top - target.height;
            break;

        case AnchorPosition.Right:
            targetLeft = anchor.right;
            targetTop = alignment === AnchorAlignment.Start ?
                anchor.top :
                anchor.bottom - target.height;
            break;

        case AnchorPosition.Bottom:
            targetLeft = alignment === AnchorAlignment.Start ?
                anchor.left :
                anchor.right - target.width;
            targetTop = anchor.bottom;
            break;
    }

    // Now, we must ensure there are no overflows
    // For width, we simply adjust the new left (we will not shrink width)
    if (targetLeft < container.left) {
        targetLeft = container.left;
    }
    else if ((targetLeft + target.width) > container.right) {
        targetLeft = container.right - target.width;
    }

    // Adjust height as appropriate, taking into consideration
    // whether we allow overlaps or not
    if (targetTop < container.top) {
        targetTop = container.top;
        targetBottom = targetTop + target.height;

        // Determine what to use to limit the bottom of the target
        // and adjust it if needed
        let bottomLimit = position === AnchorPosition.Top && !allowOverlap ?
            anchor.top : container.bottom;
        if (targetBottom > bottomLimit) {
            targetBottom = bottomLimit;
        }
    }
    else if ((targetTop + target.height) > container.bottom) {
        targetBottom = container.bottom;
        targetTop = targetBottom - target.height;

        // Now limit the top of the target if needed
        let topLimit = position === AnchorPosition.Bottom && !allowOverlap ?
            anchor.bottom : container.top;
        if (targetTop < topLimit) {
            targetTop = topLimit;
        }
    }
    else {
        targetBottom = targetTop + target.height;
    }

    // Create client rect for positioned/adjusted target
    return {
        left: targetLeft,
        right: targetLeft + target.width,
        width: target.width,
        top: targetTop,
        bottom: targetBottom,
        height: targetBottom - targetTop
    };
}

/**
 * Given a target and anchor for positioning within a container, determine if the passed in
 * position would allow placement of the target completely. Will return the amount of free
 * vertical space left after placement
 * @param target Element to be placed
 * @param anchor Element around which the target will be placed
 * @param container Area inside of which the positioned target must be
 * @param position Position around the anchor to compute
 */
function computeVerticalFit(target: ClientRect, anchor: ClientRect, container: ClientRect, position: AnchorPosition): number {
    let availableArea = getAvailableArea(anchor, container, position);

    // Width must always fit
    // Height can be variable, so we return the overflow of the target's height, which
    // can be used to choose the best possible candidate
    if (target.width <= availableArea.width) {
        return availableArea.height - target.height;
    }

    // When width does not fit, we just mark the overflow as infinite to ensure this
    // position is never a best candidate
    return -Infinity;
}

/**
 * Given a container rectangle, compute the width and height of available space for placement
 * of an element around another rectangle: the anchor
 * @param anchor The element around which placement will be done
 * @param container Total available area where placement will happen. Should contain the anchor
 * @param position Where placemente will happen with relation to the anchor
 */
function getAvailableArea(anchor: ClientRect, container: ClientRect, position: AnchorPosition): ClientArea {
    switch (position) {
        case AnchorPosition.Top:
            return {
                width: container.width,
                height: anchor.top - container.top
            };

        case AnchorPosition.Right:
            return {
                width: container.right - anchor.right,
                height: container.height
            };

        case AnchorPosition.Bottom:
            return {
                width: container.width,
                height: container.bottom - anchor.bottom
            };

        case AnchorPosition.Left:
            return {
                width: anchor.left - container.left,
                height: container.height
            };
    }
}

/**
 * Given the optimal ClientRect in global coordiantes for a target, actually place said
 * target in that position by setting its top and left values accordingly.
 * @param target The Element being positioned
 * @param finalRect ClientRect containing information on the final position of the target
 * such that it is optimal. Given in global coordinates.
 * @param setHeight Flag that indicates whether to also set the target's size based on the
 * finalRect's values (HEIGHT only)
 */
function applyOptimalPosition(target: HTMLElement, finalRect: ClientRect, setHeight?: boolean): void {
    if (target) {
        // Determine which element acts as the positioning parent for the target.
        // If the target has 'position: fixed', it will be the window. Otherwise,
        // it will be the first ancestor of the target with 'postion: !static'
        let positioningParent: Element | undefined;
        let targetStyle = getComputedStyle(target);
        if (targetStyle.position !== 'fixed') {
            positioningParent = traverseDomTree(
                target,
                el => el !== target && (el as any) !== document && getComputedStyle(el).position !== 'static'
            ) || document.documentElement || undefined;
        }

        // Set top and left of the body to our computed values, using the parent's rect.
        // If we did not set the parent above, the ClientRect will be all zeroes, which
        // works fine with 'position: fixed' since that is already in terms of the window
        let parentRect = getClientRect(positioningParent);
        target.style.left = `${finalRect.left - parentRect.left}px`;
        target.style.top = `${finalRect.top - parentRect.top}px`;
        target.style.right = 'auto';
        target.style.bottom = 'auto';

        // If we are also adusting the height of the container, we need to set overflow
        // to scrolling too.
        if (setHeight) {
            target.style.height = `${finalRect.height}px`;
            target.style.overflow = 'auto';
        }
        else {
            target.style.height = target.style.overflow = "";
        }
    }
}

/**
 * Obtain the ClientRect for a given element and include any hidden width or height (scrollable)
 * @param element The element to compute the ClientRect for
 */
function getFullClientRect(element: HTMLElement | undefined): ClientRect {
    let clientRect = getClientRect(element);

    // Add scrollWidth and scrollHeight to the rect
    if (element) {
        clientRect = {
            left: clientRect.left,
            right: clientRect.left + element.scrollWidth,
            top: clientRect.top,
            bottom: clientRect.top + element.scrollHeight,
            width: element.scrollWidth,
            height: element.scrollHeight
        };
    }

    return clientRect;
}

/**
 * Determine if the passed in element is 'position: fixed' or withing
 * a fixed position DOM tree
 * @param element Element to evaluate
 */
function isFixedElement(element: HTMLElement): boolean {
    let result = false;
    traverseDomTree(element, el => {
        // Stop processing if we get to a non-element object
        if (!isElement(el)) {
            return true;
        }

        let style = getComputedStyle(el);
        if (style.position === 'fixed') {
            result = true;
            return true;
        }
        return el === document.documentElement;
    });

    return result;
}
