import { ObjectAssign, hasOwn } from '@mecontrol/web-inline';

const nonWhiteSpaceRegex = /[^\s]+/;

/**
 * Determine whether a passed in value is either an empty string or
 * a string with only whitespace.
 * @param value Value to test
 */
function isEmptyString(value: any): boolean {
    switch (typeof value) {
        case 'string':
            // For strings, we check if it is the empty string and whether
            // it has at least one non-whitespace character
            return value === '' || !nonWhiteSpaceRegex.test(value);
        default:
            return false;
    }
}

/**
 * Version of assign() that ignores 'undefined' and 'null' properties, as well as empty
 * or whitespace strings.
 * @param target Object that will have its propertes expanded/overwritten by the merge
 * @param sources Array of objects from which to expand the target
 */
export const merge: ObjectAssign = (target: any, ...sources: any[]) => {
    for (let source of sources) {
        for (let prop in source) {
            if (hasOwn(source, prop)) {
                let propValue = source[prop];
                if (propValue !== null && propValue !== undefined && !isEmptyString(propValue)) {
                    target[prop] = propValue;
                }
            }
        }
    }

    return target;
};
