import { setQueryParams, urlEncodeArr } from './url';
import {
    ME,
    // @ts-ignore
    Promise
} from '@mecontrol/web-inline';
import { AccountType, IAccount } from '@mecontrol/public-api';
import { DataState, GraphResource, StringMap } from '@mecontrol/common';

export interface IPKCEcode {
    verifier: string;
    challenge: string;
}
/**
 * The Query Parameters required to request Code
 */
export interface CodeQParams {
    client_id: string;
    response_type: string;
    redirect_uri: string;
    scope: string;
    response_mode: string;
    state: string;
    code_challenge: string;
    code_challenge_method: string;
    prompt: string;
    login_hint?: string;
    sid?: string;
}
/**
 * The Query Parameters required to redeem code for Token
 */
export interface TokenQParams {
    client_id: string;
    scope: string;
    redirect_uri: string;
    grant_type: string;
    code_verifier: string;
}
/**
 * The request object iframe sends to parent to init Token request
 */
export interface FrameInitRequest {
    /**
     * State passed into IdP and from IdP redirect.
     *
     * Must validate against `OAuth2Handler.state` to avoid cx attacks */
    state: string;
    status: DataState;
    payload: any;
}
/**
 * The object iframe requires to redeem Code for Token
 */
export interface OAuthFrameInitRes {
    accountId: string;
    /** Will be verified by frame to proceed with token request */
    state: string;
    qParams: TokenQParams;
    graphResources: GraphResource[]
}

export class MSGraphOAuth2Builder {
    public code: IPKCEcode = { challenge: '', verifier: '' };
    public client_id: string = ME.Config.graphinfo.graphclientid;
    public scope: string = ME.Config.graphinfo.graphscope;
    public redirect_uri: string;
    public code_challenge_method: string = 'S256';
    public grant_type: string = 'authorization_code';
    public response_type: string = 'code';
    public response_mode: string = 'fragment';
    public prompt: string = 'none';

    constructor(
        public account: IAccount,
        public accountId: string,
        public graphResources: GraphResource[],
        public state?: string,
        redirect_uri?: string
    ) {
        this.code_challenge_method;
        this.response_type = 'code';
        this.response_mode = 'fragment';
        this.redirect_uri =
            redirect_uri ?? ME.Config.graphinfo.graphredirecturi;
    }
    /**
     *
     * @returns Promise resolves to string url to request code
     */
    public initCodeFlow(): Promise<void> {
        return generatePKCE().then(pkceCode => {
            this.code = pkceCode;
            this.state = this.state ?? createNewStateGuid();
        });
    }

    public getCodeReqUrl(): string {
        return setQueryParams(
            ME.Config.graphinfo.graphcodeurl,
            this.GetCodeQueryParams() as unknown as Partial<StringMap>,
            false
        );
    }

    private GetCodeQueryParams(): CodeQParams {
        let codeQParams: CodeQParams = {
            client_id: this.client_id,
            scope: this.scope,
            code_challenge: this.code?.challenge,
            code_challenge_method: this.code_challenge_method,
            prompt: this.prompt,
            redirect_uri: this.redirect_uri,
            response_mode: this.response_mode,
            response_type: this.response_type,
            state: this.state ?? createNewStateGuid()
        };
        if (this.account.type === AccountType.AAD && this.account.sessionId) {
            codeQParams.sid = this.account.sessionId;
        } else if (this.account.login_hint) {
            codeQParams.login_hint = this.account.login_hint;
        } else if (this.account.memberName) {
            codeQParams.login_hint = this.account.memberName;
        } else {
            throw 'Fatal Error: Account is missing critical id info.';
        }
        return codeQParams;
    }
    public GetFrameInitResponse(): OAuthFrameInitRes {
        if (!this.state) {
            throw 'Critical error: Missing state of transaction.';
        }
        return {
            accountId: this.accountId,
            state: this.state,
            qParams: {
                code_verifier: this.code.verifier,
                client_id: this.client_id,
                scope: this.scope,
                redirect_uri: this.redirect_uri,
                grant_type: this.grant_type
            },
            graphResources: this.graphResources
        };
    }
}
/**
 * Randomly generate a user state to go across OAuth Flow
 * for profile picture
 * @returns The randomly generated user state guid string
 */
export function createNewStateGuid(): string {
    let guidHolder = 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx';
    let hex = '0123456789abcdef';
    let r = 0;
    let guidResponse = '';
    for (let i = 0; i < 36; i++) {
        if (guidHolder[i] !== '-' && guidHolder[i] !== '4') {
            // each x and y needs to be random
            r = (Math.random() * 16) | 0;
        }
        if (guidHolder[i] === 'x') {
            guidResponse += hex[r];
        } else if (guidHolder[i] === 'y') {
            // clock-seq-and-reserved first hex is filtered and remaining hex values are random
            r &= 0x3; // bit and with 0011 to set pos 2 to zero ?0??
            r |= 0x8; // set pos 3 to 1 as 1???
            guidResponse += hex[r];
        } else {
            guidResponse += guidHolder[i];
        }
    }
    return guidResponse;
}

async function generatePKCE(): Promise<IPKCEcode> {
    var crypto = window.crypto;
    // check for win32 msCrypto library instead
    if (window.msCrypto && !crypto) {
        crypto = window.msCrypto;
    }
    let verifier = generateCodeVerifier(crypto);
    let challenge = await generateCodeChallenge(crypto, verifier);
    return {
        verifier,
        challenge
    };
}

function generateCodeVerifier(crypto: Crypto): string {
    var RANDOM_BYTE_ARR_LENGTH = 32;

    const buffer = new Uint8Array(RANDOM_BYTE_ARR_LENGTH);
    crypto.getRandomValues(buffer);

    var pkceCodeVerifierB64 = urlEncodeArr(buffer);
    return pkceCodeVerifierB64;
}

export async function generateCodeChallenge(
    crypto: Crypto,
    code_verifier: string
): Promise<string> {
    var codeArrayBuffer = stringToUtf8Arr(code_verifier);
    if (!!window.crypto) {
        return await getSubtleCryptoDigest(codeArrayBuffer);
    } else {
        return new Promise<string>((resolve, reject) => {
            if (crypto.subtle) {
                var cryptoOp = crypto.subtle.digest('SHA-256', codeArrayBuffer);
                // @ts-ignore
                cryptoOp.oncomplete = function (e) {
                    var code_challenge_arr = e.target.result;
                    resolve(urlEncodeArr(new Uint8Array(code_challenge_arr)));
                };
            }
        });
    }
    function getSubtleCryptoDigest(data: Uint8Array) {
        if (window.crypto.subtle) {
            return window.crypto.subtle
                .digest('SHA-256', data)
                .then(function (code_challenge_arr) {
                    return urlEncodeArr(new Uint8Array(code_challenge_arr));
                });
        }
        return Promise.resolve('');
    }
}

function stringToUtf8Arr(sDOMStr: string): Uint8Array {
    var nChr;
    var nArrLen = 0;
    var nStrLen = sDOMStr.length;
    /* mapping... */
    for (var nMapIdx = 0; nMapIdx < nStrLen; nMapIdx++) {
        nChr = sDOMStr.charCodeAt(nMapIdx);
        if (nChr < 0x80) {
            nArrLen += 1;
        } else if (nChr < 0x800) {
            nArrLen += 2;
        } else if (nChr < 0x10000) {
            nArrLen += 3;
        } else if (nChr < 0x200000) {
            nArrLen += 4;
        } else if (nChr < 0x4000000) {
            nArrLen += 5;
        } else {
            nArrLen += 6;
        }
    }
    var aBytes = new Uint8Array(nArrLen);
    /* transcription... */
    for (var nIdx = 0, nChrIdx = 0; nIdx < nArrLen; nChrIdx++) {
        nChr = sDOMStr.charCodeAt(nChrIdx);
        if (nChr < 128) {
            /* one byte */
            aBytes[nIdx++] = nChr;
        } else if (nChr < 0x800) {
            /* two bytes */
            aBytes[nIdx++] = 192 + (nChr >>> 6);
            aBytes[nIdx++] = 128 + (nChr & 63);
        } else if (nChr < 0x10000) {
            /* three bytes */
            aBytes[nIdx++] = 224 + (nChr >>> 12);
            aBytes[nIdx++] = 128 + ((nChr >>> 6) & 63);
            aBytes[nIdx++] = 128 + (nChr & 63);
        } else if (nChr < 0x200000) {
            /* four bytes */
            aBytes[nIdx++] = 240 + (nChr >>> 18);
            aBytes[nIdx++] = 128 + ((nChr >>> 12) & 63);
            aBytes[nIdx++] = 128 + ((nChr >>> 6) & 63);
            aBytes[nIdx++] = 128 + (nChr & 63);
        } else if (nChr < 0x4000000) {
            /* five bytes */
            aBytes[nIdx++] = 248 + (nChr >>> 24);
            aBytes[nIdx++] = 128 + ((nChr >>> 18) & 63);
            aBytes[nIdx++] = 128 + ((nChr >>> 12) & 63);
            aBytes[nIdx++] = 128 + ((nChr >>> 6) & 63);
            aBytes[nIdx++] = 128 + (nChr & 63);
        } /* if (nChr <= 0x7fffffff) */ else {
            /* six bytes */
            aBytes[nIdx++] = 252 + (nChr >>> 30);
            aBytes[nIdx++] = 128 + ((nChr >>> 24) & 63);
            aBytes[nIdx++] = 128 + ((nChr >>> 18) & 63);
            aBytes[nIdx++] = 128 + ((nChr >>> 12) & 63);
            aBytes[nIdx++] = 128 + ((nChr >>> 6) & 63);
            aBytes[nIdx++] = 128 + (nChr & 63);
        }
    }
    return aBytes;
}
